package cn.patterncat.tracing.config;

import brave.SpanCustomizer;
import cn.patterncat.tracing.ExtraTracingProperties;
import cn.patterncat.tracing.component.HttpBodyTraceHelper;
import cn.patterncat.tracing.component.servlet.MultiReadableHttpServletRequest;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

/**
 * Created by cat on 2018-12-04.
 */
@Configuration
@EnableConfigurationProperties(ExtraTracingProperties.class)
public class ExtraTagInterceptor implements HandlerInterceptor, WebMvcConfigurer {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExtraTagInterceptor.class);

    private final ExtraTracingProperties properties;

    public ExtraTagInterceptor(ExtraTracingProperties properties) {
        this.properties = properties;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        SpanCustomizer span = (SpanCustomizer) request.getAttribute(SpanCustomizer.class.getName());

        span.tag(properties.getServletRequestQueryTag(),Objects.toString(request.getQueryString(),""));

        if(properties.isExtraServletHeaderEnabled()){
            if(properties.getTraceTagHeaderMappings() != null && !properties.getTraceTagHeaderMappings().isEmpty()){
                properties.getTraceTagHeaderMappings()
                        .entrySet()
                        .stream()
                        .forEach(entry -> {
                            String tagName = entry.getKey();
                            String headerNames = entry.getValue();
                            if(StringUtils.hasText(tagName) || StringUtils.hasText(headerNames)){
                                LOGGER.error("detect invalid traceTagHeaderMappings: {}",entry);
                                return ;
                            }
                            Optional<String> result = Arrays.stream(headerNames.split(properties.getTagMultiHeaderSeparator()))
                                    //NOTE 这里可能返回null
                                    .map(name -> Objects.toString(request.getHeader(name),""))
                                    .filter(e -> StringUtils.isEmpty(e) == false)
                                    .findFirst();

                            span.tag(tagName,result.orElse(""));
                        });
            }
        }

        if(properties.isTraceServletRequestBody() && request instanceof MultiReadableHttpServletRequest && HttpBodyTraceHelper.isContentTypePrintable(properties.getPrintableMediaSubTypes(),request.getContentType())){
            MultiReadableHttpServletRequest wrapper = (MultiReadableHttpServletRequest)request;
            try {
                byte[] data = IOUtils.toByteArray(wrapper.getInputStream());
                String requestBody = HttpBodyTraceHelper.getContentAsString(data, properties.getMaxPrintBodyBytes(), request.getCharacterEncoding());
                span.tag(properties.getServletRequestBodyTag(),requestBody);
                LOGGER.debug("request body:{}",requestBody);
            } catch (IOException e) {
                LOGGER.error(e.getMessage(),e);
            }

        }
        return true;
    }

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(this);
    }
}
