package cn.opencodes.framework.tools.utils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 文件操作类
 * 可在生产环境（Web容器）下和开发环境下保持一致的使用方法
 * 对于相对路径，可通过setBasePath方法指定前缀
 * 开发环境下的相对路径其前缀为用户当前目录：System.getProperty("user.dir")
 * 生产环境（Web容器）下的相对路径其前缀为应用的根目录：servletContextEvent.getServletContext().getRealPath("/")
 * @author HJ
 */
public class FileUtils extends org.apache.commons.io.FileUtils{
    private static final Logger LOG = LoggerFactory.getLogger(FileUtils.class);
        
    /**
     * 工具类，无实例对象，仅仅提供静态方法
     */
    private FileUtils(){};
    
    private static String basePath;
    
    
    /**
     * 系统的默认目录为当前用户目录，可通过此函数重新设置
     * @param basePath 相对路径前缀
     */
    public static void setBasePath(String basePath) {
        FileUtils.basePath = basePath.replace("\\", "/");
    }
    
    /**
     * 获取 web根目录或当前工作目录或指定目录 下面的资源的绝对路径
     * / 返回 web根目录或当前工作目录或指定目录 的绝对路径
     * /WEB-INF 返回 web根目录或当前工作目录或指定目录 下的WEB-INF 的绝对路径
     * WEB-INF 返回 web根目录或当前工作目录或指定目录 下的WEB-INF 的绝对路径
     * 
     * 特殊：对windows下的绝对路径不做转换    
     * c:/test/test.txt 返回 c:/test/test.txt
     * 
     * @param path 相对于WEB根目录的路径
     * @return 绝对路径
     */
    public static String getAbsolutePath(String path){
        //在windows下，如果路径包含：,为绝对路径，则不进行转换
        if(path.contains(":")){
            return path.replace("\\", "/");
        }        
        path=path.replace("\\", "/").trim();        
        path=basePath+"/"+path;
        while(path.contains("//")){
            path=path.replace("//", "/");
        }
        return path;
    }
    
    /**
     * 追加写入文件 -- 另起一行
     * @param path 文件绝对路径
     * @param text 要追加的文件内容
     * @return 是否追加成功
     */
    public static boolean appendTextInNewLine(String path,String text){
        return appendText(path,text+System.getProperty("line.separator"));
    }
    /**
     * 追加写入文件
     * @param path 文件绝对路径
     * @param text 要追加的文件内容
     * @return 是否追加成功
     */
    public static boolean appendText(String path,String text){
        try{
            File file=new File(path);
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file,true), "utf-8"))) {
                writer.write(text);
                writer.close();
            }
            return true;
        }catch(IOException e){
            LOG.error("写文件出错",e);
        }
        return false;
    }
   
    
    /**
     * 文件复制
     * @param inFile 输入文件
     * @param outFile 输出文件
     */
    public static void copyFile(File inFile, File outFile){
        try {
            copyFile(new FileInputStream(inFile),outFile);
        } catch (FileNotFoundException ex) {
            LOG.error("文件不存在",ex);
        }
    }
    /**
     * 把输入流中的内容拷贝到一个文件
     * @param in 输入流
     * @param outFile 文件对象
     */
    public static void copyFile(InputStream in, File outFile){
        OutputStream out = null;
        try {
            byte[] data=readAll(in);
            out = new FileOutputStream(outFile);
            out.write(data, 0, data.length);
            out.close();
        } catch (Exception ex) {
            LOG.error("文件操作失败",ex);
        } finally {
            try {
                if(in!=null){
                    in.close();
                }
            } catch (IOException ex) {
             LOG.error("文件操作失败",ex);
            }
            try {
                if(out!=null){
                    out.close();
                }
            } catch (IOException ex) {
             LOG.error("文件操作失败",ex);
            }
        }
    }
    /**
     * 读取一个文件的所有字节
     * @param file 文件对象
     * @return 字节数组
     */
    public static byte[] readAll(File file){
        try {
            return readAll(new FileInputStream(file));
        } catch (Exception ex) {
            LOG.error("读取文件失败",ex);
        }
        return null;
    }
    /**
     * 从输入流中读取所有字节
     * @param in 输入流
     * @return 字节数组
     */
    public static byte[] readAll(InputStream in){
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            byte[] buffer = new byte[1024];
            for (int n ; (n = in.read(buffer))>0 ; ) {
                out.write(buffer, 0, n);
            }
        } catch (IOException ex) {
            LOG.error("读取文件失败",ex);
        }
        return out.toByteArray();
    }
    /**
     * 获取一个文件的输入流
     * @param path 相对路径或绝对路径
     * @return 文件输入流
     */
    public static FileInputStream getInputStream(String path) {
        try {
            return new FileInputStream(getAbsolutePath(path));
        } catch (FileNotFoundException ex) {
            LOG.error("文件没有找到",ex);
        }
        return null;
    }
    
    public static InputStream getInputStreamByClass(String path) {
    	return FileUtils.class.getResourceAsStream(path);
    }
    
    /**
     * 判断一个文件是否存在
     * @param path 相对路径或绝对路径
     * @return 是否存在
     */
    public static boolean existsFile(String path){
        try{
            File file=new File(getAbsolutePath(path));
            if(file.exists()){
                return true;
            }
        }catch(Exception ex){
            LOG.error("文件操作失败",ex);
        }
        return false;
    }
    /**
     * 把字节内容写入新文件
     * @param path 相对路径或绝对路径
     * @param data 字节数组
     * @return 新文件
     */
    public static File createAndWriteFile(String path, byte[] data){
        try{
            File file=new File(getAbsolutePath(path));
            if(!file.getParentFile().exists()){
                file.getParentFile().mkdirs();
            }
            file.createNewFile();
            try (OutputStream out = new FileOutputStream(file)) {
                out.write(data, 0, data.length);
            }
            return file;
        }catch(Exception ex){
            LOG.error("文件操作失败",ex);
        }
        return null;
    }
    /**
     * 把文本内容写入一个新文件
     * 如果文件已经存在
     * 则覆盖
     * @param path 相对路径或绝对路径
     * @param text 文本
     * @return 文件
     */
    public static File createAndWriteFile(String path, String text){
        try{
            File file=new File(getAbsolutePath(path));
            if(!file.getParentFile().exists()){
                file.getParentFile().mkdirs();
            }
            file.createNewFile();
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file),"utf-8"))) {
                writer.write(text);
            }
            return file;
        }catch(Exception ex){
            LOG.error("文件操作失败",ex);
        }
        return null;
    }
    /**
     * 删除文件
     * @param path 相对路径或绝对路径
     * @return 是否成功
     */
    public static boolean removeFile(String path){
        try{
            File file=new File(getAbsolutePath(path));
            if(file.exists()){
                if(!file.delete()){
                    file.deleteOnExit();
                }
            }
            return true;
        }catch(Exception ex){
            LOG.error("文件操作失败",ex);
        }
        return false;
    }
    
    /**
     *  删除当前目录下面的文件
     */
 	public static void deleteFiles(String workspaceRootPath) {
 		File file = new File(workspaceRootPath);
 		if (file.exists()) {
 			if (file.isDirectory()) {
 				File[] files = file.listFiles();
 				for (int i = 0; i < files.length; i++) {
 					files[i].delete();
 				}
 			}
 		}
 	}
    
    /**
     * 获取文本内容
     * @param path 相对路径或绝对路径
     * @return 行的列表
     */
    public static List<String> getTextFileContent(String path) {
        try {
            return getTextFileContent(new FileInputStream(getAbsolutePath(path)));
        } catch (FileNotFoundException ex) {
            if(!path.contains("apdplat.licence")){
                LOG.error("文件不存在", ex);
            }
        }
        //Null Object设计模式
        return Collections.emptyList();
    }
    /**
     * 获取输入流中的文本内容
     * @param in 文本文件输入流
     * @return 行的列表
     */
    public static List<String> getTextFileContent(InputStream in) {
        List<String> result = new ArrayList<>();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(in, "utf-8"))) {
            for (;;) {
                String line = reader.readLine();
                if (line == null)
                    break;
                result.add(line);
            }
        } catch(Exception e){
            LOG.error("读取文件失败",e);
        }
        return result;
    }
    
    /**
     * 获取文件大小
     * @param file
     */
    public static long getFileSize(File file) {
        if (file.exists() && file.isFile()) {
           return file.length();
        }
        return 0;
    }
    
    /**  
 	 * <p>Title: convertFileSize</p>  
 	 * <p>Description: 设置上传文件大小</p>  
 	 * @param size
 	 */  
 	public static String convertFileSize(long size) {
         long kb = 1024;
         long mb = kb * 1024;
         long gb = mb * 1024;
         if (size >= gb) {
             return String.format("%.1f GB", (float) size / gb);
         } else if (size >= mb) {
             float f = (float) size / mb;
             return String.format(f > 100 ? "%.0f MB" : "%.1f MB", f);
         } else if (size >= kb) {
             float f = (float) size / kb;
             return String.format(f > 100 ? "%.0f KB" : "%.1f KB", f);
         } else {
             return String.format("%d B", size);
         }
    }
 	
 	/**
	 * 判断操作系统是windsow
	 */
	public static boolean isWindowsOS(){
		String os = System.getProperty("os.name");
		if (StringUtils.isNotBlank(os) && os.startsWith("Windows")) {
			return true;
		}
		return false;
	}
}