package cn.omisheep.commons.util;

import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 用于生成定时任务
 *
 * @author zhouxinchen[1269670415@qq.com]
 * @version 1.0.0
 * @since 1.0.0
 */
public abstract class TaskBuilder {

    /**
     * 缓存任务计数
     */
    private static final AtomicInteger taskNumber = new AtomicInteger(1);

    /**
     * 定时器
     */
    private static ScheduledExecutorService timer;

    static {
        create();
    }

    private TaskBuilder() {
    }

    public static ScheduledFuture<?> schedule(Runnable task, String periodTimeVal) {
        return schedule(task, TimeUtils.parseTimeValue(periodTimeVal), TimeUnit.MILLISECONDS);
    }

    public static ScheduledFuture<?> schedule(Runnable task, long period, TimeUnit timeUnit) {
        if (period == 0) return null;
        return timer.scheduleAtFixedRate(task, 0, period, timeUnit);
    }

    public static ScheduledFuture<?> scheduleDelay(Runnable task, String delayTimeVal, String periodTimeVal) {
        return scheduleDelay(task, TimeUtils.parseTimeValue(delayTimeVal), TimeUtils.parseTimeValue(periodTimeVal), TimeUnit.MILLISECONDS);
    }

    public static ScheduledFuture<?> scheduleDelay(Runnable task, long delay, long period, TimeUnit timeUnit) {
        if (period == 0) return scheduleOnceDelay(task, delay, timeUnit);
        return timer.scheduleAtFixedRate(task, delay, period, timeUnit);
    }

    public static ScheduledFuture<?> scheduleOnce(Runnable task) {
        return timer.schedule(task, 0, TimeUnit.MICROSECONDS);
    }

    public static ScheduledFuture<?> scheduleOnceDelay(Runnable task, String delayTimeVal) {
        return scheduleOnceDelay(task, TimeUtils.parseTimeValue(delayTimeVal), TimeUnit.MILLISECONDS);
    }

    public static ScheduledFuture<?> scheduleOnceDelay(Runnable task, long delay, TimeUnit timeUnit) {
        if (delay == 0) return scheduleOnce(task);
        return timer.schedule(task, delay, timeUnit);
    }

    /**
     * 创建定时器
     */
    public static void create() {
        if (null != timer) {
            shutdownNow();
        }
        timer = new ScheduledThreadPoolExecutor(1,
                r -> newThread(r, StringUtils.format("Timer-{}", taskNumber.getAndIncrement()))
        );
    }

    /**
     * 销毁全局定时器
     */
    public static void shutdown() {
        if (null != timer) {
            timer.shutdown();
        }
    }

    /**
     * 销毁全局定时器
     *
     * @return 销毁时未被执行的任务列表
     */
    public static List<Runnable> shutdownNow() {
        if (null != timer) {
            return timer.shutdownNow();
        }
        return null;
    }

    private static Thread newThread(Runnable runnable, String name) {
        final Thread t = new Thread(null, runnable, name);
        t.setDaemon(false);
        return t;
    }

}
