package cn.omisheep.autt.core.browser;

import cn.omisheep.autt.*;
import cn.omisheep.autt.core.ContentType;
import cn.omisheep.autt.core.Cookies;
import cn.omisheep.autt.core.Method;
import cn.omisheep.autt.core.RequestHeader;
import cn.omisheep.autt.util.ConvertUtil;
import cn.omisheep.autt.util.URLParser;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static cn.omisheep.autt.util.ConvertUtil.objectToForm;
import static cn.omisheep.autt.util.ConvertUtil.objectToJSON;

public class BrowserRequest implements GetRequest, PostRequest {

    private       int                 connTimeout;
    private       int                 readTimeout;
    private       Charset             charset = StandardCharsets.UTF_8;
    private       Object              body;
    private final Method              method;
    private final String              domain;
    private final String              uri;
    private final String              path;
    private final Map<String, String> param   = new HashMap<>();
    private final RequestHeader       header  = new RequestHeader();
    private final Cookies             cookies = new Cookies();
    private final BrowserConnect      connect;

    BrowserRequest(BrowserConnect connect, String url, Method method) {
        this.method  = method;
        this.connect = connect;
        URLParser.URLInfo urlInfo = URLParser.parse(url);
        this.domain = urlInfo.getDomain();
        this.uri    = urlInfo.getUri();
        this.path   = urlInfo.getPath();
        this.param.putAll(urlInfo.getParam());
        List<Cookies> cookies = connect.getCookiesListByUri(this.domain, this.uri);
        if (cookies != null) {
            this.cookies.putList(cookies);
        }
        this.connTimeout = connect.connTimeout;
        this.readTimeout = connect.readTimeout;
    }

    @Override
    public Response send() {
        try {
            HttpURLConnection conn = requestConfig();
            List<String>      list = conn.getHeaderFields().get("Set-Cookie");
            Cookies           coo  = new Cookies();
            if (list != null) {
                for (String s : list) {
                    Cookies  c       = new Cookies();
                    String[] split   = s.replaceAll(" ", "").split(";");
                    String   _domain = domain;
                    String   _path   = uri;
                    for (int i = 0; i < split.length; i++) {
                        String[] s1 = split[i].split("=");
                        if (i == 0) {
                            c.put(s1[0], s1[1]);
                            continue;
                        }
                        if (s1[0].equals("Domain")) {
                            _domain = s1[1];
                            continue;
                        }
                        if (s1[0].equals("Path")) {
                            _path = s1[1];
                        }
                    }
                    coo.putAll(c);
                    this.connect.setCookie(_domain, _path, c);
                }
            }

            return new BrowserResponse(this.connect, conn.getHeaderFields(), readResponseBody(conn), coo, conn.getResponseCode());
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public Connect getConnect() {
        return this.connect;
    }

    @Override
    public Request cookie(String key, String val) {
        this.cookies.put(key, val);
        return this;
    }

    @Override
    public Request cookies(Cookies cookies) {
        this.cookies.putAll(cookies);
        return this;
    }

    @Override
    public Request header(String key, String val) {
        header.put(key, val);
        return this;
    }

    @Override
    public Request header(Map<String, String> header) {
        this.header.putAll(header);
        return this;
    }

    @Override
    public Request userAgent(String userAgent) {
        return header(RequestHeader.USER_AGENT, userAgent);
    }

    @Override
    public PostRequest body(Object o, ContentType contentType) {
        this.body = o;
        return contentType(contentType);
    }

    @Override
    public PostRequest body(Object o) {
        this.body = o;
        return this;
    }

    @Override
    public PostRequest contentType(ContentType contentType) {
        header.setContentType(contentType);
        return this;
    }

    @Override
    public GetRequest param(Object param) {
        if (param == null) return this;
        ConvertUtil.objectToMap(param, this.param);
        return this;
    }

    @Override
    public GetRequest param(String key, Object val) {
        if (val != null) param.put(key, val.toString());
        else param.put(key, "");
        return this;
    }

    @Override
    public Request connTimeout(int connTimeout) {
        this.connTimeout = connTimeout;
        return this;
    }

    @Override
    public Request readTimeout(int readTimeout) {
        this.readTimeout = readTimeout;
        return this;
    }

    @Override
    public Request charset(String charset) {
        this.charset = Charset.forName(charset);
        return this;
    }

    @Override
    public Request charset(Charset charset) {
        this.charset = charset;
        return this;
    }

    @Override
    public String toString() {
        return send().toString();
    }

    private HttpURLConnection openConnection() throws IOException {
        URL realUrl = new URL(getURL());
        return (HttpURLConnection) realUrl.openConnection();
    }

    private HttpURLConnection requestConfig() throws IOException {
        HttpURLConnection conn = openConnection();
        conn.setConnectTimeout(connTimeout);
        conn.setReadTimeout(readTimeout);
        for (Map.Entry<String, String> entry : header.entrySet()) {
            conn.setRequestProperty(entry.getKey(), entry.getValue());
        }
        conn.setRequestProperty("Cookie", cookies.toPropertyString());
        if (header.getContentType() != null) {
            String type = header.getContentType().getCt();
            if (this.charset != null) {
                type += ";charset=" + this.charset;
            }
            conn.setRequestProperty("Content-Type", type);
        }

        conn.setRequestMethod(method.name());
        if (method.hasBody()) {
            String _body = parseBody();
            conn.setDoOutput(true);
            conn.setDoInput(true);
            conn.setRequestProperty("Content-Length", _body.length() + "");
            PrintWriter out = new PrintWriter(conn.getOutputStream());
            out.print(_body);
            out.flush();
            out.close();
        }
        return conn;
    }

    private String readResponseBody(HttpURLConnection conn) throws IOException {
        StringBuilder  resultBody = new StringBuilder();
        BufferedReader in;
        try {
            in = new BufferedReader(new InputStreamReader(conn.getInputStream(), charset));
        } catch (IOException e) {
            in = new BufferedReader(new InputStreamReader(conn.getErrorStream(), charset));
        }
        String line;
        while ((line = in.readLine()) != null) {
            resultBody.append(line).append("\n");
        }
        in.close();
        if (resultBody.length() > 0) resultBody.deleteCharAt(resultBody.length() - 1);
        return resultBody.toString();
    }

    private String parseBody() {
        if (this.body == null) {
            return "";
        }

        ContentType contentType = header.getContentType();

        if (contentType == null) {
            try {
                throw new Exception("Content-Type is null");
            } catch (Exception e) {
                e.printStackTrace();
            }
            return "";
        }

        if (this.body instanceof String) {
            return (String) this.body;
        }

        try {
            if (contentType.equals(ContentType.JSON)) {
                return objectToJSON(this.body);
            } else if (contentType.equals(ContentType.X_WWW_FORM_URLENCODED)) {
                return objectToForm(this.body);
            } else {
                throw new Exception("content-type mismatch");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    private String getURL() {
        if (param.isEmpty()) return path;
        return path + "?" + ConvertUtil.objectToForm(param);
    }

}
