package cn.nxtools.common.base;

import java.util.function.Function;

/**
 * @author niuchangqing
 * object相关工具方法
 */
public final class Objects {

    /**
     * 对象是否为空
     * @param obj   对象
     * @return      true or false
     */
    public static boolean isNull(Object obj) {
        return obj == null;
    }

    /**
     * 对象非空
     * @param obj       对象
     * @return          true or false
     */
    public static boolean nonNull(Object obj) {
        return obj != null;
    }

    /**
     * 对象为空即返回默认值,否则返回当前对象
     * @param obj           指定对象
     * @param defaultValue  obj参数为空,返回的参数
     * @param <T>           T
     * @return              返回obj or defaultValue
     */
    public static <T> T defaultIfNull(T obj, T defaultValue) {
        return obj == null ? defaultValue : obj;
    }

    /**
     * 比较俩个参数是否一样 <br>
     * o1不等于null, 调用 {@link Object#equals(Object)}
     * @param o1            参数1
     * @param o2            参数2
     * @return              true or false
     * @since               1.0.3
     */
    public static boolean equals(Object o1, Object o2) {
        if (o1 == o2) {
            return true;
        }
        return o1 != null && o1.equals(o2);
    }

    /**
     * 参数为空返回null，非空执行用户自定义处理逻辑
     * <p>
     *     {@code User user = null;}<br>
     *     {@code String name = Objects.map(user, u -> u.getName());}<br>
     *     {@code System.out.println(name);}<br>
     *     输出: null
     * </p>
     * @param obj               参数
     * @param mapper            参数非空自定义执行逻辑
     * @param <T>               参数T
     * @param <U>               返回结果U
     * @return                  返回结果
     * @since 1.0.5
     */
    public static <T, U> U map(T obj, Function<? super T, ? extends U> mapper) {
        return map(obj, mapper, null);
    }

    /**
     * 参数为空返回自定义的value，非空执行用户自定义处理逻辑
     * <p>
     *     {@code User user = new User();}<br>
     *     {@code user.setName("lisi");}<br>
     *     {@code String name = Objects.map(user, u -> u.getName(), "zhangsan")}<br>
     *     {@code System.out.println(name);}<br>
     *     输出: lisi
     * </p>
     * @param obj               参数
     * @param mapper            参数非空自定义执行逻辑
     * @param defaultValue      参数为空默认返回的值
     * @param <T>               参数T
     * @param <U>               返回U
     * @return                  返回值
     * @since 1.0.5
     */
    public static <T, U> U map(T obj, Function<? super T, ? extends U> mapper, U defaultValue) {
        Preconditions.checkNotNull(mapper, "mapper must not be null");
        if (obj == null) {
            return defaultValue;
        }
        return mapper.apply(obj);
    }
}
