/*
 * This file is part of ELCube.
 *
 * ELCube is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ELCube is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with ELCube.  If not, see <https://www.gnu.org/licenses/>.
 */
package cn.nkpro.elcube.components.pipeline.cards

import cn.nkpro.elcube.annotation.NkNote
import cn.nkpro.elcube.co.NkAsyncInvoker
import cn.nkpro.elcube.co.spel.NkSpELManager
import cn.nkpro.elcube.docengine.EnumDocClassify
import cn.nkpro.elcube.docengine.NkAbstractCard
import cn.nkpro.elcube.docengine.model.DocDefIV
import cn.nkpro.elcube.docengine.model.DocHV
import cn.nkpro.elcube.docengine.model.DocPipeline
import cn.nkpro.elcube.security.NkSecurityRunner
import cn.nkpro.elcube.security.SecurityUtilz
import cn.nkpro.elcube.security.bo.UserDetails
import com.alibaba.fastjson.JSON
import org.apache.commons.lang3.StringUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.annotation.Order
import org.springframework.expression.EvaluationContext
import org.springframework.stereotype.Component

import java.lang.reflect.UndeclaredThrowableException
import java.util.concurrent.Future
import java.util.concurrent.TimeUnit
import java.util.stream.Collectors

@Order(2001)
@NkNote("服务调用")
@Component("NkCardPipelineCapture")
class NkCardPipelineCapture extends NkAbstractCard<Data,Def> {

    @Override
    boolean supports(String classify) {
        return classify == EnumDocClassify.PIPELINE.name()
    }

    @Autowired
    private NkSpELManager spELManager
    @Autowired
    private NkAsyncInvoker asyncInvoker
    @Autowired@SuppressWarnings("all")
    private NkSecurityRunner securityRunner

    @Override
    Data afterGetData(DocHV doc, Data data, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline && d.resultSpEL){
            doc.getResults().put(defIV.cardKey, data.result)
        }
        return super.afterGetData(doc, data, defIV, d) as Data
    }

    @Override
    Data afterUpdated(DocHV doc, Data data, Data original, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline && d.resultSpEL){
            doc.getResults().put(defIV.cardKey, data.result)
        }
        return super.afterUpdated(doc, data, original, defIV, d) as Data
    }

    @Override
    Data calculate(DocHV doc, Data data, DocDefIV defIV, Def d, boolean isTrigger, Object options) {

        def context = spELManager.createContext(doc)

        // 如果先决条件不满足，直接返回
        if(StringUtils.isNotBlank(d.preconditionSpEL)){
            if(!spELManager.invoke(d.preconditionSpEL, context)){
                return null
            }
        }
        data = invokeService(d.services, context, d.parallel)

        if(doc instanceof DocPipeline && d.resultSpEL){
            data.result = JSON.parse(spELManager.convert(d.resultSpEL, spELManager.createContext(data)))
            doc.getResults().put(defIV.cardKey, data)
        }

        return data
    }

    Data invokeService(List<Service> services, EvaluationContext context, Boolean parallel){

        List<Service> clones = new ArrayList<>(
                services.findAll {
                srv->
                    if(srv.preconditionSpEL){
                        return spELManager.invoke(srv.preconditionSpEL, context)
                    }
                    return true
            }
        )

        if(parallel){

//            def user = SecurityUtilz.user
//
//            clones.parallelStream()
//                .map({ route ->
//                    securityRunner.returnAsUser(user, true,{ ->
//                        spELManager.invoke(route.invokeSpEL, context)
//                    })
//                }).collect(Collectors.toList())

            Map<Service,Future> futures = new HashMap<>()
            clones.forEach{ route ->
                futures.put(
                        route,
                        asyncInvoker.async(
                                SecurityUtilz.user,
                                { ->
                                    return spELManager.invoke(route.invokeSpEL, context)
                                }
                        )
                )
            }
            Data data = new Data()
            data.services = new HashMap<>()
            try{
                futures.forEach{ route, future ->
                    ServiceResult routeResult = new ServiceResult()
                    routeResult.response = future.get(30, TimeUnit.SECONDS)
                    if(routeResult.response && route.mappingSpEL){
                        routeResult.mapping = JSON.parse(spELManager.convert(route.mappingSpEL, routeResult.response))
                    }
                    data.services.put(route.key,routeResult)
                }
            }catch(RuntimeException e){
                // 因为是异步方法，所以需要取出被包装了的异常信息
                if(e instanceof UndeclaredThrowableException){
                    throw e.getUndeclaredThrowable().getCause()
                }
                throw e
            }
            return data
        }else{

            Data data = new Data()
            data.services = new HashMap<>()
            clones.forEach({ route ->
                ServiceResult routeResult = new ServiceResult()
                routeResult.response = spELManager.invoke(route.invokeSpEL, context)
                if(route.mappingSpEL){
                    routeResult.mapping = JSON.parse(spELManager.convert(route.mappingSpEL, routeResult.response))
                }
                data.services.put(route.key,routeResult)
            })
            return data
        }
    }


    static class Def implements Serializable{
        String preconditionSpEL
        Boolean parallel
        String resultSpEL
        List<Service> services
    }

    static class Service implements Serializable{
        String key
        String preconditionSpEL
        String invokeSpEL
        String desc
        String mappingSpEL
    }

    static class Data implements Serializable{
        Map<String,ServiceResult> services
        Object result
    }

    static class ServiceResult implements Serializable{
        Object response
        Object mapping
    }
}
