/*
 * This file is part of ELCube.
 *
 * ELCube is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ELCube is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with ELCube.  If not, see <https://www.gnu.org/licenses/>.
 */
package cn.nkpro.elcube.components.pipeline.cards

import cn.nkpro.elcube.annotation.NkNote
import cn.nkpro.elcube.co.spel.NkSpELManager
import cn.nkpro.elcube.docengine.EnumDocClassify
import cn.nkpro.elcube.docengine.NkAbstractCard
import cn.nkpro.elcube.docengine.model.DocDefIV
import cn.nkpro.elcube.docengine.model.DocHV
import cn.nkpro.elcube.docengine.model.DocPipeline
import org.apache.commons.lang3.StringUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

import java.util.regex.Pattern


@Order(12)
@NkNote("流水线")
@Component("NkCardHeaderPipeline")
class NkCardHeaderPipeline extends NkAbstractCard<Data,Def> {

    private static Pattern pattern = Pattern.compile("((\\d+)Y)?((\\d+)M)?((\\d+)D)?((\\d+)h)?((\\d+)m)?((\\d+)s)?")

    @Override
    boolean supports(String classify) {
        return StringUtils.equals(classify, EnumDocClassify.PIPELINE.name())
    }

    @Autowired
    private NkSpELManager spELManager

    @Override
    String getPosition() {
        return POSITION_HEADER
    }

    @Override
    Data afterGetData(DocHV doc, Data data, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline){
            doc.expire = data.expire
        }
        return super.afterGetData(doc, data, defIV, d) as Data
    }

    @Override
    Data afterUpdated(DocHV doc, Data data, Data original, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline){
            doc.expire = data.expire
        }
        return super.afterUpdated(doc, data, original, defIV, d) as Data
    }

    @Override
    Data calculate(DocHV doc, Data data, DocDefIV defIV, Def d, boolean isTrigger, Object options) {

        data.expire = null
        if(d.expire){
            def matcher = pattern.matcher(d.expire)
            if(matcher.matches()){

                def calendar = Calendar.getInstance()

                def Y = matcher.group(2)
                if(Y)calendar.add(Calendar.YEAR,Y as Integer)

                def M = matcher.group(4)
                if(M)calendar.add(Calendar.MONTH,M as Integer)

                def D = matcher.group(6)
                if(D)calendar.add(Calendar.DAY_OF_MONTH,D as Integer)

                def h = matcher.group(8)
                if(h)calendar.add(Calendar.HOUR_OF_DAY,h as Integer)

                def m = matcher.group(10)
                if(m)calendar.add(Calendar.MINUTE,m as Integer)

                def s = matcher.group(12)
                if(s)calendar.add(Calendar.SECOND,s as Integer)

                data.expire = (Long) (calendar.getTimeInMillis() / 1000)

                if(doc instanceof DocPipeline){
                    doc.expire = data.expire
                }
            }
        }

        if(d.docNameDefaultSpEL && !doc.docName){
            doc.docName = spELManager.invoke(d.docNameDefaultSpEL, doc)
        }

        return super.calculate(doc, data, defIV, d, isTrigger, options) as Data
    }



    static class Def implements Serializable{
        String docNameDefaultSpEL
        String expire
    }

    static class Data implements Serializable{
        Long expire
    }
}
