/*
 * This file is part of ELCube.
 *
 * ELCube is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ELCube is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with ELCube.  If not, see <https://www.gnu.org/licenses/>.
 */
package cn.nkpro.elcube.components.pipeline.cards

import cn.nkpro.elcube.annotation.NkNote
import cn.nkpro.elcube.co.spel.NkSpELManager
import cn.nkpro.elcube.docengine.EnumDocClassify
import cn.nkpro.elcube.docengine.NkAbstractCard
import cn.nkpro.elcube.docengine.model.DocDefIV
import cn.nkpro.elcube.docengine.model.DocHV
import cn.nkpro.elcube.docengine.model.DocPipeline
import cn.nkpro.elcube.exception.NkSystemException
import com.alibaba.fastjson.JSON
import org.apache.commons.lang3.StringUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.annotation.Order
import org.springframework.expression.EvaluationContext
import org.springframework.stereotype.Component

import java.util.stream.Collectors

@Order(2002)
@NkNote("服务路由")
@Component("NkCardPipelineRouter")
class NkCardPipelineRouter extends NkAbstractCard<Data,Def> {

    @Override
    boolean supports(String classify) {
        return classify == EnumDocClassify.PIPELINE.name()
    }

    @Autowired
    private NkSpELManager spELManager

    @Override
    Data afterGetData(DocHV doc, Data data, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline && d.resultSpEL){
            doc.getResults().put(defIV.cardKey, data.result)
        }
        return super.afterGetData(doc, data, defIV, d) as Data
    }

    @Override
    Data afterUpdated(DocHV doc, Data data, Data original, DocDefIV defIV, Def d) {
        if(doc instanceof DocPipeline && d.resultSpEL){
            doc.getResults().put(defIV.cardKey, data.result)
        }
        return super.afterUpdated(doc, data, original, defIV, d) as Data
    }

    @Override
    Data calculate(DocHV doc, Data data, DocDefIV defIV, Def d, boolean isTrigger, Object options) {

        def context = spELManager.createContext(doc)

        // 如果先决条件不满足，直接返回
        if(StringUtils.isNotBlank(d.preconditionSpEL)){
            if(!spELManager.invoke(d.preconditionSpEL, context)){
                return null
            }
        }

        data = invokeRoute(d.routes, context)

        if(doc instanceof DocPipeline && d.resultSpEL){
            data.result = JSON.parse(spELManager.convert(d.resultSpEL, spELManager.createContext(data)))
            doc.getResults().put(defIV.cardKey, data)
        }

        return data
    }

    Data invokeRoute(List<Route> routes, EvaluationContext context){

        if(routes){
            List<Route> clones = new ArrayList<>(routes)
            Exception exception = null
            while(true){
                if(!clones) {
                    throw new NkSystemException("流水线接口调用失败:"+exception?.message)
                }
                int sum = clones.stream().collect(Collectors.summingInt({ r -> r.weight }))
                long random = Math.random()*sum as Long

                Route route = null
                long value = 0
                for(Route r : clones){
                    value += r.weight?:0
                    if(value > random){// 因为随机数范围为[0,1)，所以这里不能等于
                        route = r
                        break
                    }
                }

                try{
                    Data data = new Data()
                    data.key = route.key
                    data.response = spELManager.invoke(route.invokeSpEL, context)

                    if(route.successSpEL){
                        if(!spELManager.invoke(route.successSpEL,data.response)){
                            clones.remove(route)
                            continue
                        }
                    }

                    if(route.mappingSpEL){
                        data.mapping = JSON.parse(spELManager.convert(route.mappingSpEL, data.response))
                    }

                    return data
                }catch(Exception e){
                    log.error(e.getMessage(),e)
                    clones.remove(route)
                    exception = e
                }
            }
        }
    }


    static class Def implements Serializable{
        String preconditionSpEL
        String resultSpEL
        List<Route> routes
    }

    static class Route implements Serializable{
        String key
        String invokeSpEL
        String successSpEL
        String desc
        Integer weight
        String mappingSpEL
    }

    static class Data implements Serializable{
        String key
        Object response
        Object mapping
        Object result
    }
}
