/*
 * This file is part of ELCube.
 *
 * ELCube is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ELCube is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with ELCube.  If not, see <https://www.gnu.org/licenses/>.
 */
package cn.nkpro.elcube.components.pipeline.cards

import cn.nkpro.elcube.annotation.NkNote
import cn.nkpro.elcube.co.spel.NkSpELManager
import cn.nkpro.elcube.docengine.EnumDocClassify
import cn.nkpro.elcube.docengine.NkAbstractCard
import cn.nkpro.elcube.docengine.model.DocDefIV
import cn.nkpro.elcube.docengine.model.DocHV
import cn.nkpro.elcube.docengine.model.DocPipeline
import cn.nkpro.elcube.docengine.service.NkPipelineService
import cn.nkpro.elcube.exception.NkInputFailedCaution
import org.apache.commons.lang3.StringUtils
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.annotation.Order
import org.springframework.expression.EvaluationContext
import org.springframework.stereotype.Component
import org.springframework.util.DigestUtils

import javax.servlet.http.HttpServletRequest

/**
 * 流水线入参卡片
 *
 * 负责接收流水线参数的输入并校验
 *
 * 并将参数md5后作为业务主键
 */
@Order(2001)
@NkNote("入参")
@Component("NkCardPipelineInputs")
class NkCardPipelineInputs extends NkAbstractCard<Map,Def> {

    @Override
    boolean supports(String classify) {
        return classify == EnumDocClassify.PIPELINE.name()
    }

    @Autowired
    private NkSpELManager spELManager

    @Override
    Map afterCreate(DocHV doc, DocHV preDoc, Map data, DocDefIV defIV, Def d) {

        Map<String,String> map = NkPipelineService.request.get()?:Collections.emptyMap() as Map<String, String>

        // 设置输入参数
        if(d.params){
            def context = spELManager.createContext(doc)
            context.setVariable("\$PREV",preDoc)
            d.params.forEach{ item ->
                if(map.containsKey(item.name)){
                    data.put(item.name,StringUtils.trimToEmpty(map.get(item.name)))
                }else if(StringUtils.isNotBlank(item.defaultValueSpEL)){
                    data.put(item.name,StringUtils.trimToEmpty(spELManager.invoke(item.defaultValueSpEL,context) as String))
                }
            }
        }

        // 将参数设置到流水线对象
        if(doc instanceof DocPipeline){
            doc.setInputs(data)
        }

        // 设置业务主键
        if(StringUtils.isNotBlank(doc.getDef().getBusinessKeySpEL())) {
            Object businessKey = spELManager.invoke(doc.getDef().getBusinessKeySpEL(), spELManager.createContext(doc))
            doc.setBusinessKey((businessKey as String)?:'')
        }else{
            doc.setBusinessKey(DigestUtils.md5DigestAsHex(data.toMapString().getBytes()))
        }

        return super.afterCreate(doc, preDoc, data, defIV, d) as Map
    }

    @Override
    Map calculate(DocHV doc, Map data, DocDefIV defIV, Def d, boolean isTrigger, Object options) {
        if(doc instanceof DocPipeline){
            // 校验输入参数
            if(d.params){
                d.params.forEach{ item ->
                    def value = data.get(item.name)
                    if(item.required && value==null){
                        throw new NkInputFailedCaution(String.format("参数[%s]不能为空",item.name))
                    }
                }
            }
            doc.setInputs(data)
        }

        // 设置业务主键
        if(StringUtils.isNotBlank(doc.getDef().getBusinessKeySpEL())) {
            Object businessKey = spELManager.invoke(doc.getDef().getBusinessKeySpEL(), spELManager.createContext(doc))
            doc.setBusinessKey((businessKey as String)?:'')
        }else{
            doc.setBusinessKey(DigestUtils.md5DigestAsHex(data.toMapString().getBytes()))
        }

        return super.calculate(doc, data, defIV, d, isTrigger, options) as Map
    }

    static class Def implements Serializable{
        List<Param> params
    }

    static class Param{
        String name
        String desc
        Boolean required
        String defaultValueSpEL
    }
}
