package cn.nikeo.permisos.compiler

import com.google.auto.common.MoreTypes
import com.google.common.base.Joiner
import com.google.common.base.Preconditions
import com.google.common.collect.ImmutableSet
import com.google.common.collect.MoreCollectors.toOptional
import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.asClassName
import com.squareup.kotlinpoet.asTypeName
import java.util.Optional
import javax.lang.model.element.AnnotationMirror
import javax.lang.model.element.Element
import javax.lang.model.element.TypeElement
import javax.lang.model.type.TypeKind

/**
 * Static helper methods for writing a processor.
 */
object Processors {
    /**
     * Returns the name of a class, including prefixing with enclosing class names. i.e. for inner
     * class Foo enclosed by Bar, returns Bar_Foo instead of just Foo
     */
    private fun getEnclosedName(name: ClassName): String {
        return Joiner.on('_').join(name.simpleNames)
    }

    /**
     * Returns an equivalent class name with the `.` (dots) used for inner classes replaced with
     * `_`.
     */
    fun getEnclosedClassName(className: ClassName): ClassName {
        return ClassName(className.packageName, getEnclosedName(className))
    }

    /**
     * Prepends the given string to the beginning of the class name.
     */
    fun prepend(
        name: ClassName,
        prefix: String
    ): ClassName {
        return name.peerClass(prefix + name.simpleName)
    }

    /**
     * @return true if element inherits directly or indirectly from the className
     */
    fun isAssignableFrom(element: TypeElement, className: ClassName): Boolean {
        return isAssignableFromAnyOf(element, ImmutableSet.of(className))
    }

    /**
     * @return true if element inherits directly or indirectly from any of the classNames
     */
    private fun isAssignableFromAnyOf(
        element: TypeElement,
        classNames: ImmutableSet<ClassName>
    ): Boolean {
        for (className in classNames) {
            @Suppress("DEPRECATION")
            if (element.asClassName() == className) {
                return true
            }
        }
        val superClass = element.superclass
        // None type is returned if this is an interface or Object
        // Error type is returned for classes that are generated by this processor
        if (superClass.kind != TypeKind.NONE && superClass.kind != TypeKind.ERROR) {
            Preconditions.checkState(superClass.kind == TypeKind.DECLARED)
            if (isAssignableFromAnyOf(MoreTypes.asTypeElement(superClass), classNames)) {
                return true
            }
        }
        for (iface in element.interfaces) {
            // Skip errors and keep looking. This is especially needed for classes generated by this
            // processor.
            if (iface.kind == TypeKind.ERROR) {
                continue
            }
            Preconditions.checkState(
                iface.kind == TypeKind.DECLARED,
                "Interface type is %s",
                iface.kind
            )
            if (isAssignableFromAnyOf(MoreTypes.asTypeElement(iface), classNames)) {
                return true
            }
        }
        return false
    }

    /**
     * Returns true if the given element has an annotation with the given class name.
     */
    fun hasAnnotation(element: Element, className: ClassName): Boolean {
        return getAnnotationMirrorOptional(element, className).isPresent
    }

    /**
     * Returns the annotation mirror from the given element that corresponds to the given class.
     *
     * @throws [IllegalArgumentException] if 2 or more annotations are found.
     * @return [Optional.empty] if no annotation is found on the element.
     */
    fun getAnnotationMirrorOptional(
        element: Element,
        className: ClassName
    ): Optional<AnnotationMirror> {
        return element.annotationMirrors.stream()
            .filter { mirror: AnnotationMirror ->
                @Suppress("DEPRECATION")
                mirror.annotationType.asTypeName() == className
            }
            .collect(toOptional())
    }

    /**
     * Returns the annotation mirror from the given element that corresponds to the given class.
     *
     * @throws IllegalStateException if the given element isn't annotated with that annotation.
     */
    fun getAnnotationMirror(
        element: Element,
        className: ClassName
    ): AnnotationMirror {
        val annotationMirror: Optional<AnnotationMirror> =
            getAnnotationMirrorOptional(
                element,
                className
            )
        return if (annotationMirror.isPresent) {
            annotationMirror.get()
        } else {
            throw IllegalStateException(
                String.format(
                    "Couldn't find annotation %s on element %s. Found annotations: %s",
                    className, element.simpleName, element.annotationMirrors
                )
            )
        }
    }
}
