package org.jsmth.jorm.service;

import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;
import org.jsmth.jorm.jdbc.SchemaUpdateStrategy;
import org.jsmth.jorm.jdbc.SlaveJdbcDao;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * 读写分离的entityDao基类，自动创建master和slave表，并只对slave表创建索引
 * User: mason
 * Date: 2010-1-29
 * Time: 13:31:06
 */
@SuppressWarnings({"JavaDoc", "UnusedDeclaration"})
public abstract class ReadWriteEntityDao<KEY extends Serializable, MODEL extends Identifier<KEY>> extends EntityDao<KEY, MODEL> {

    protected JdbcDao slaveJdbcDao;

    public ReadWriteEntityDao(Class<MODEL> entityClass) {
        super(entityClass);
    }

    public ReadWriteEntityDao(Class<MODEL> entityClass, JdbcDao masterJdbcDao, JdbcDao slaveJdbcDao) {
        super(entityClass);
        setJdbcDao( masterJdbcDao);
        setSlaveJdbcDao( slaveJdbcDao);
    }

    //override

    @Override
    public void init() throws SmthDataAccessException {
        Assert.notNull(entityClass, "entityClass must be set!");
        Assert.notNull(getMasterDao(), "master jdbcDao must be set!");
        Assert.notNull(getSlaveJdbcDao(), "slave jdbcDao must be set!");

        getJdbcDao().setEntityEventCallback(this);
        getSlaveJdbcDao().setEntityEventCallback(this);
        if (updateSchemaWhileInitial) {
            if ((getSlaveJdbcDao() instanceof SlaveJdbcDao) && ((SlaveJdbcDao) getSlaveJdbcDao()).isCluster()) {
                //如果说是真正的主库，则不能自动创建表，必须显示的调用api去创建，其它的仍然自动创建表
            } else {
                //本地环境或者根本还没用集群从库，则仍然需呀主库创建索引
                getMasterDao().updateSchema(entityClass, true);
            }
        }
    }

    /**
     * 对于主从分离，需要在具体的entityDao上进行rebuildSchema的操作
     *
     * @param entityClass d
     */
    @SuppressWarnings({"unchecked"})
    public void rebuildSchema(Class... entityClass) throws SmthDataAccessException {
        for (Class clazz : entityClass) {
            getMasterDao().updateSchema(clazz, SchemaUpdateStrategy.CREATE, false);
            getSlaveJdbcDao().doCreateIndex(clazz, true);
        }
    }

    @Override
    public List<KEY> findIds(String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().findIds(entityClass, where, params);
    }

    public List<KEY> findIds(int limit, boolean asceding) throws SmthDataAccessException {
        if (limit < 1) limit = 1;
        String sql = String.format("1=1 order by id %s limit %d", (asceding ? "asc" : "desc"), limit);
        return getSlaveJdbcDao().findIds(entityClass, sql);
    }

    @Override
    public List<MODEL> findAll() throws SmthDataAccessException {
        return getSlaveJdbcDao().findAll(entityClass);
    }

    /**
     * 根据一组条件，查询得到一组model
     *
     * @param where d
     * @param params d
     * @return 返回信息 d
     */
    @Override
    public List<MODEL> findModels(String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().find(entityClass, where, params);
    }

    @Override
    public int countModels(String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().count(entityClass, where, params);
    }

    @Override
    public <T, E> List<Map.Entry<E, Integer>> groupCountColumn(Class<E> columnClass, String fieldName, String countFieldName, String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().groupCountColumn(entityClass, columnClass, fieldName, countFieldName, where, params);
    }

    @Override
    public <T, E> List<Map.Entry<E, Integer>> groupMaxColumn(Class<E> columnClass, String fieldName, String maxFieldName, String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().groupMaxColumn(entityClass, columnClass, fieldName, maxFieldName, where, params);
    }

    @Override
    public <T, E> List<Map.Entry<E, Integer>> groupMinColumn(Class<E> columnClass, String fieldName, String minFieldName, String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().groupMinColumn(entityClass, columnClass, fieldName, minFieldName, where, params);
    }

    @Override
    public <T, E> List<Map.Entry<E, Integer>> groupSumColumn(Class<E> columnClass, String fieldName, String sumFieldName, String where, Object... params) throws SmthDataAccessException {
        return getSlaveJdbcDao().groupSumColumn(entityClass, columnClass, fieldName, sumFieldName, where, params);
    }

    /**
     * 根据一组条件，查询得到一个model
     *
     * @param where d
     * @param params d
     * @return 返回信息
     */
    @Override
    public MODEL findUnique(String where, Object... params) throws SmthDataAccessException {
        List<MODEL> model = findModels(where, params);
        if (model.isEmpty()) {
            return null;
        } else {
            return model.get(0);
        }
    }

    //ext logic


    //gs


    @Override
    public void setJdbcDao(JdbcDao jdbcDao) {
        this.jdbcDao = jdbcDao;
        this.setBaseJdbcDao(jdbcDao);
    }

    public JdbcDao getMasterDao() {
        return jdbcDao;
    }

    public void setMasterDao(JdbcDao masterDao) {
        this.jdbcDao = masterDao;
        this.setBaseJdbcDao(masterDao);
    }

    public JdbcDao getSlaveJdbcDao() {
        return slaveJdbcDao;
    }

    public void setSlaveJdbcDao(JdbcDao slaveJdbcDao) {
        this.slaveJdbcDao = slaveJdbcDao;
    }
}
