package org.jsmth.jorm.service;

import org.apache.commons.lang.Validate;
import org.jsmth.exception.SmthExceptionDictHandler;
import org.jsmth.util.ValueUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.jorm.domain.IPageConditon;
import org.jsmth.jorm.jdbc.EntityEventCallback;
import org.jsmth.jorm.jdbc.Event;
import org.jsmth.domain.Identifier;
import org.jsmth.util.IdentifierKeyHelper;
import org.jsmth.jorm.query.WhereQuery;
import org.jsmth.page.TailPage;
import org.jsmth.util.PropertyUtils;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.*;

/**
 * 基于JDBC的实体缓存服务
 *
 * @author mason
 */
@SuppressWarnings({"unchecked", "UnusedDeclaration"})
public abstract class EntityService<KEY extends Serializable, MODEL extends Identifier<KEY>, DAO extends EntityDao<KEY, MODEL>> implements EntityEventCallback,IEntityService<KEY,MODEL> {
    protected EntityEventCallback entityEventCallback;
    protected Logger log = LoggerFactory.getLogger(this.getClass());

    protected DAO entityDao;

    protected ExtentServicePropertyMethod extentPropertyMethods;

    public EntityService() {
        extentPropertyMethods = new ExtentServicePropertyMethod(this);
    }

    public ExtentServicePropertyMethod extMethods() {
        return extentPropertyMethods;
    }

    @PostConstruct
    public void init() {
        Validate.notNull(entityDao, "entity dao must be set!");
        entityDao.setEntityEventCallback(this);
    }

    @Override
    public String getNameById(KEY id) {
        return getStrPropertyById(id, "name");
    }

    @Override
    public Object getPropertyById(KEY id, String propertyName) {
        return getPropertyById(id, propertyName, null);
    }

    @Override
    public Object getPropertyById(KEY id, String propertyName, Object defaultValue) {
        MODEL model = getById(id);
        if (model == null) {
            return defaultValue;
        }
        try {
            return PropertyUtils.getPropertyValue(model, propertyName);
        } catch (Exception ex) {
            ex.printStackTrace();
            return defaultValue;
        }
    }

    @Override
    public String getStrPropertyById(KEY id, String propertyName) {
        return getStrPropertyById(id, propertyName, "");
    }

    @Override
    public String getStrPropertyById(KEY id, String propertyName, String defaultValue) {
        Object value = getPropertyById(id, propertyName, defaultValue);
        return value.toString();
    }

    @Override
    public int getIntPropertyById(KEY id, String propertyName) {
        return getIntPropertyById(id, propertyName, 0);
    }

    @Override
    public int getIntPropertyById(KEY id, String propertyName, int defaultValue) {
        return ValueUtil.toInt(getPropertyById(id, propertyName, null), defaultValue);
    }

    @Override
    public long getLongPropertyById(KEY id, String propertyName) {
        return getLongPropertyById(id, propertyName, 0);
    }

    @Override
    public long getLongPropertyById(KEY id, String propertyName, long defaultValue) {
        return ValueUtil.toLong(getPropertyById(id, propertyName, null), defaultValue);
    }

    protected TailPage<MODEL> pageFindModels(WhereQuery query) {
        return entityDao.pageFindModels(query);
    }

    protected List<MODEL> findModels(WhereQuery query) {
        return entityDao.findModels(query);
    }


    protected TailPage<MODEL> pageFindModels(String where, int pageNumber, int pageSize, Object... params) {
        return entityDao.pageFindModels(where, pageNumber, pageSize, params);
    }

    protected List<MODEL> findModels(String where, Object... params) {
        return entityDao.findModels(where, params);
    }

    protected List<KEY> findIds(String where, Object... params) {
        return entityDao.findIds(where, params);
    }

    protected List<KEY> findIds(int limit, boolean asceding) {
        return entityDao.findIds(limit, asceding);
    }

    protected <CONDITION> MODEL conditionFindModel(CONDITION condition) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.getModel(sql, sqlParas.toArray());
    }

    protected <CONDITION extends IPageConditon> TailPage<MODEL> conditionPageFindModels(CONDITION condition) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.pageFindModels(sql, condition.getPageNumber(), condition.getPageSize(), sqlParas.toArray());
    }

    protected <CONDITION extends IPageConditon, COLUMN> TailPage<COLUMN> conditionPageFindColumns(CONDITION condition, Class<COLUMN> columnClass, String columnName) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.pageFindColumn(columnClass, columnName, sql, condition.getPageNumber(), condition.getPageSize(), sqlParas.toArray());
    }

    protected <CONDITION extends IPageConditon> List<MODEL> conditionFindModels(CONDITION condition) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.findModels(sql, sqlParas.toArray());
    }

    protected <CONDITION extends IPageConditon, COLUMN> List<COLUMN> conditionFindColumns(CONDITION condition, Class<COLUMN> columnClass, String columnName) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.findColumn(columnClass, columnName, sql, sqlParas.toArray());
    }

    protected <CONDITION extends IPageConditon> List<KEY> conditionFindIds(CONDITION condition) {
        List sqlParas = new ArrayList();
        String sql = buildQuery(condition, sqlParas);
        return entityDao.findIds(sql, sqlParas.toArray());
    }

    protected <CONDITION> String buildQuery(CONDITION condition, List sqlParas) {
        return "1=1";
    }

    public MODEL getById(KEY id) throws SmthDataAccessException {
        return entityDao.getById(id);
    }
    public MODEL getById(boolean checkNull,KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getById(id), checkNull);
    }
    public MODEL getById(boolean checkNull,String errorMessage,String errorCode,KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getById(id), checkNull, errorMessage, errorCode);
    }
    public MODEL getById(boolean checkNull, SmthExceptionDictHandler exceptionEnum, KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getById(id), checkNull, exceptionEnum);
    }

    public MODEL getByIdOrNew(KEY id) throws SmthDataAccessException {
        return getByIdOrNew(id, true);
    }

    public MODEL getByIdOrNew(KEY id, boolean setid) throws SmthDataAccessException {
        return getByIdOrNew(id, setid, true);
    }

    public MODEL getByIdOrNew(KEY id, boolean setid, boolean insert) throws SmthDataAccessException {
        return entityDao.getByIdOrNew(id, setid, insert);
    }

    public MODEL getOrCreateById(KEY id) throws SmthDataAccessException {
        return entityDao.getOrCreateById(id);
    }

    public List<MODEL> findAll() throws SmthDataAccessException {
        return entityDao.findAll();
    }

    public List<MODEL> findByIds(List<KEY> ids) throws SmthDataAccessException {
        return entityDao.findByIds(ids);
    }

    public List<MODEL> findByIds(Set<KEY> ids) throws SmthDataAccessException {
        return entityDao.findByIds(ids);
    }

    public List<MODEL> findByIdsOrNew(List<? extends KEY> ids) {
        return entityDao.findByIdsOrNew(ids);
    }

    public List<MODEL> findByIdsOrNew(Set<? extends KEY> ids) {
        return entityDao.findByIdsOrNew(ids);
    }

    public MODEL insert(MODEL entity) throws SmthDataAccessException {
        return entityDao.insert(entity);
    }

    public void insertAll(Collection<MODEL> entities) throws SmthDataAccessException {
        entityDao.insertAll(entities);
    }

    public int update(MODEL entity) throws SmthDataAccessException {
        return entityDao.update(entity);
    }

    public int updateFields(MODEL entity, String... fieldNames) throws SmthDataAccessException {
        return entityDao.updateFields(entity, fieldNames);
    }

    protected int updateFeilds(boolean exclude, String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        return entityDao.updateFeilds(exclude, fieldNames, values, where, params);
    }

    protected int updateFeilds(String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        return entityDao.updateFeilds(fieldNames, values, where, params);
    }

    protected int updateFeild(String fieldName, Object value, String where, Object... params) throws SmthDataAccessException {
        return entityDao.updateFeild(fieldName, value, where, params);
    }


    protected int updateAllFeilds(boolean exclude, String[] fieldNames, Object[] values) throws SmthDataAccessException {
        return entityDao.updateAllFeilds(exclude, fieldNames, values);
    }

    protected int updateAllFeilds(String[] fieldNames, Object[] values) throws SmthDataAccessException {
        return entityDao.updateAllFeilds(fieldNames, values);
    }

    protected int updateAllFeild(String fieldName, Object value) throws SmthDataAccessException {
        return entityDao.updateAllFeild(fieldName, value);
    }


    public int updateAll(Collection<MODEL> entities) throws SmthDataAccessException {
        return entityDao.updateAll(entities);
    }

    public int updateFeildsAll(Collection<MODEL> entities, String... fieldNames) throws SmthDataAccessException {
        return entityDao.updateFeildsAll(entities, fieldNames);
    }

    public MODEL save(MODEL entity) throws SmthDataAccessException {
        return getEntityDao().save(entity);
    }

    public boolean delete(MODEL entity) throws SmthDataAccessException {
        return entityDao.delete(entity);
    }

    public int deleteAll(Collection<MODEL> entities) throws SmthDataAccessException {
        return entityDao.deleteAll(entities);
    }

    public boolean deleteById(KEY id) throws SmthDataAccessException {
        return entityDao.deleteById(id);
    }

    public int deleteByIds(Collection<? extends KEY> ids) throws SmthDataAccessException {
        return entityDao.deleteByIds(ids);
    }

    protected int deleteModels(String where, Object... params) throws SmthDataAccessException {
        return entityDao.deleteModels(where, params);
    }


    // gs

    public DAO getEntityDao() {
        return entityDao;
    }

    public void setEntityDao(DAO entityDao) {
        this.entityDao = entityDao;
    }

    public List getFieldsByCollection(Collection beans, String fieldName) {
        return IdentifierKeyHelper.getFields(beans, fieldName);
    }

    public <T> void copyProperties(T orig, T dest) throws Exception {
        copyProperties(orig, dest);
    }


    public MODEL getModel(boolean exclude, String[] fieldNames, String where, Object... params) throws SmthDataAccessException {
        return this.entityDao.getModel(exclude, fieldNames, where, params);
    }

    public MODEL getModel(String where, Object... params) throws SmthDataAccessException {
        return this.entityDao.getModel(where, params);
    }

    public MODEL getModel(boolean checkNull,boolean exclude, String[] fieldNames, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull);
    }

    public MODEL getModel(boolean checkNull,String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull);
    }

    public MODEL getModel(boolean checkNull ,String errorMessage,String errorCode,boolean exclude, String[] fieldNames, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull, errorMessage, errorCode);
    }

    public MODEL getModel(boolean checkNull ,String errorMessage,String errorCode,String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull, errorMessage, errorCode);
    }

    public MODEL getModel(boolean checkNull , SmthExceptionDictHandler exceptionEnum, boolean exclude, String[] fieldNames, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull, exceptionEnum);
    }

    public MODEL getModel(boolean checkNull , SmthExceptionDictHandler exceptionEnum, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull, exceptionEnum);
    }



    public <E> E getColumn(Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return this.entityDao.getColumn(columnClass, fieldName, where, params);
    }
    public <E> E getColumn(boolean checkNull,Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull);
    }
    public <E> E getColumn(boolean checkNull ,String errorMessage,String errorCode,Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull, errorMessage, errorCode);
    }
    public <E> E getColumn(boolean checkNull , SmthExceptionDictHandler exceptionEnum, Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull, exceptionEnum);
    }

    public MODEL getSortFirstOne(String fieldName, boolean desc) {
        return this.entityDao.getSortFirstOne(fieldName, desc);
    }

    public MODEL getFirstModel() {
        return this.entityDao.getFirstModel();
    }

    public MODEL getLastModel() {
        return this.entityDao.getLastModel();
    }


    @Override
    public void setBeginKey(KEY beginKey) {
        setBeginKey(beginKey,true);
    }
    @Override
    public void setBeginKey(KEY beginKey,boolean isdelete) {
        this.entityDao.setBeginKey(beginKey,isdelete,entityDao.getEntityClass());
    }

    @Override
    public void batchInvokeEvent(Collection targets, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.batchInvokeEvent(targets, event);
        }
    }

    @Override
    public void invokeEvent(Object target, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.invokeEvent(target, event);
        }
    }

    public EntityEventCallback getEntityEventCallback() {
        return entityEventCallback;
    }

    public void setEntityEventCallback(EntityEventCallback entityEventCallback) {
        this.entityEventCallback = entityEventCallback;
    }

    public void copyProperties(Object sourceObject, Object targetObject, boolean copyNullProperty) {
        PropertyUtils.copyProperties(sourceObject, targetObject, copyNullProperty);
    }

}
