package org.jsmth.jorm.service;

import org.apache.commons.beanutils.PropertyUtils;
import org.jsmth.util.IdentifierKeyHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.exception.SmthExceptionDictHandler;
import org.jsmth.jorm.jdbc.BaseJdbcDao;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;
import org.jsmth.jorm.jdbc.Table;
import org.jsmth.jorm.query.WhereQuery;
import org.jsmth.page.TailPage;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.*;

/**
 * 基于JDBC的实体Dao，
 *
 * @author mason
 */
@SuppressWarnings({"UnusedDeclaration", "JavaDoc"})
public abstract class EntityDao<KEY extends Serializable, MODEL extends Identifier<KEY>> extends BaseEntityDao<MODEL> {

    protected Logger log = LoggerFactory.getLogger(this.getClass());

    protected boolean updateSchemaWhileInitial = true;

    protected JdbcDao jdbcDao;

    protected Class<MODEL> entityClass;

    public EntityDao(Class<MODEL> entityClass) {
        super(entityClass);
        this.entityClass = entityClass;
    }

    public EntityDao(Class<MODEL> entityClass, JdbcDao jdbcDao) {
        super(entityClass, jdbcDao);
        this.entityClass = entityClass;
    }

    @Override
    public void setBaseJdbcDao(BaseJdbcDao baseJdbcDao) {
        super.setBaseJdbcDao(baseJdbcDao);
        this.jdbcDao = (JdbcDao) baseJdbcDao;
    }

    public void setJdbcDao(JdbcDao jdbcDao) {
        setBaseJdbcDao(jdbcDao);
    }

    public JdbcDao getJdbcDao() {
        return jdbcDao;
    }

    public Class<MODEL> getEntityClass() {
        return entityClass;
    }

    @PostConstruct
    public void init() throws SmthDataAccessException {
        Assert.notNull(entityClass, "entityClass must be set!");
        Assert.notNull(getJdbcDao(), "jdbcDao must be set!");
        if (updateSchemaWhileInitial) {
            getJdbcDao().updateSchema(entityClass);
        }
    }

    //<editor-fold desc="rebuildSchema">
    @SuppressWarnings({"unchecked"})
    public void rebuildSchema(Class... entityClass) throws SmthDataAccessException {
        rebuildSchema(true,entityClass);
    }
    public void rebuildSchema(boolean createIndex,Class... entityClass) throws SmthDataAccessException {
        //和init方法中一样，重建表也是先重建从库，并指定主库不创建索引
        for (Class clazz : entityClass) {
            getJdbcDao().doCreateTable(clazz, true);
            if(createIndex) {
                getJdbcDao().doCreateIndex(clazz, true);
            }
        }
    }
    public <K extends Serializable> void setBeginKey(K beginKey,boolean isdelete,Class... entityClasss) {
        for (Class clazz : entityClasss) {
            try {
                Identifier<K> instance = (Identifier<K>)clazz.newInstance();
                instance.setIdentifier(beginKey);
                instance=getJdbcDao().insert(instance);
                if(isdelete) {
                    getJdbcDao().delete(instance);
                }
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }
    }
    //</editor-fold>

    //<editor-fold desc="getId">
    public MODEL getById(KEY id) throws SmthDataAccessException {
        return getJdbcDao().getById(entityClass, id);
    }
    public MODEL getById(boolean checkNull,KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getJdbcDao().getById(checkNull,entityClass, id), checkNull);
    }
    public MODEL getById(boolean checkNull,String errorMessage,String errorCode,KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getById(checkNull,id), checkNull, errorMessage, errorCode);
    }
    public MODEL getById(boolean checkNull, SmthExceptionDictHandler exceptionEnum, KEY id) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getById(checkNull,id), checkNull, exceptionEnum);
    }
    public MODEL getByIdOrNew(KEY id) throws SmthDataAccessException {
        return getByIdOrNew(id, true);
    }
    public MODEL getByIdOrNew(KEY id, boolean setid) throws SmthDataAccessException {
        return getByIdOrNew(id, setid, true);
    }
    public MODEL getByIdOrNew(KEY id, boolean setid, boolean insert) throws SmthDataAccessException {
        MODEL model = getJdbcDao().getById(entityClass, id);
        if (model == null) {
            try {
                model = this.entityClass.newInstance();
            } catch (Exception e) {
                e.printStackTrace();
                return null;
            }
            if (setid)
                model.setIdentifier(id);
            if (insert) {
                model = insert(model);
            }
        }
        return model;
    }
    public MODEL getOrCreateById(KEY id) throws SmthDataAccessException {
        return getJdbcDao().getOrCreateById(entityClass, id);
    }
    //</editor-fold>

    //<editor-fold desc="get">
    public MODEL getSortFirstOne(String fieldName, boolean desc) {
        String where = "1=1 order by " + fieldName;
        if (desc)
            where += " DESC";
        where += " limit 1,1";
        return getModel(where);
    }

    public MODEL getFirstModel() {
        return getSortFirstOne("id", false);
    }

    public MODEL getLastModel() {
        return getSortFirstOne("id", true);
    }

    //</editor-fold>
    //<editor-fold desc="find ids">
    public List<MODEL> findByIds(List<? extends KEY> ids) throws SmthDataAccessException {
        if(ids==null || ids.size()==0){
            return new ArrayList<>();
        }
        return getJdbcDao().findByIds(entityClass, ids, true);
    }

    public List<MODEL> findByIds(Set<? extends KEY> ids) throws SmthDataAccessException {
        if(ids==null || ids.size()==0){
            return new ArrayList<>();
        }
        return getJdbcDao().findByIds(entityClass, ids, true);
    }

    public List<MODEL> findByIdsOrNew(List<? extends KEY> ids) throws SmthDataAccessException {
        if(ids==null || ids.size()==0){
            return new ArrayList<>();
        }
        List<MODEL> dbModels = getJdbcDao().findByIds(entityClass, ids, true);
        List<MODEL> models = new LinkedList<MODEL>(dbModels);
        List<KEY> noKeys = new LinkedList<KEY>(ids);
        for (MODEL model : dbModels) {
            noKeys.remove(model.getIdentifier());
        }
        for (KEY noKey : noKeys) {
            models.add(getByIdOrNew(noKey));
        }
        return models;
    }

    public List<MODEL> findByIdsOrNew(Set<? extends KEY> ids) throws SmthDataAccessException {
        if(ids==null || ids.size()==0){
            return new ArrayList<>();
        }
        List<MODEL> dbModels = getJdbcDao().findByIds(entityClass, ids, true);
        List<MODEL> models = new LinkedList<MODEL>(dbModels);
        List<KEY> noKeys = new LinkedList<KEY>(ids);
        for (MODEL model : dbModels) {
            noKeys.remove(model.getIdentifier());
        }
        for (KEY noKey : noKeys) {
            models.add(getByIdOrNew(noKey));
        }
        return models;
    }
    //</editor-fold>

    //<editor-fold desc="save">
    public MODEL save(MODEL entity) throws SmthDataAccessException {
        return getJdbcDao().save(entity);
    }
    //</editor-fold>

    //<editor-fold desc="insert">
    public MODEL insert(MODEL entity) throws SmthDataAccessException {
        return getJdbcDao().insert(entity);
    }

    public void insertAll(Collection<MODEL> entities) throws SmthDataAccessException {
        getJdbcDao().insertAll(entities);
    }
    //</editor-fold>

    //<editor-fold desc="update">
    public int update(MODEL entity) throws SmthDataAccessException {
        return getJdbcDao().updateModel(entity);
    }

    public int updateFields(MODEL entity, String... fieldNames) throws SmthDataAccessException {
        return getJdbcDao().updateFeilds(entity, fieldNames);
    }

    public int updateFeilds(boolean exclude, String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().updateFeilds(entityClass, exclude, fieldNames, values, where, params);
    }

    public int updateFeilds(String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().updateFeilds(entityClass, false, fieldNames, values, where, params);
    }

    public int updateFeild(String fieldName, Object value, String where, Object... params) throws SmthDataAccessException {
        Object[] values = new Object[1];
        values[0] = value;
        return updateFeilds(new String[]{fieldName}, values, where, params);
    }


    public int updateAllFeilds(boolean exclude, String[] fieldNames, Object[] values) throws SmthDataAccessException {
        return updateFeilds(exclude, fieldNames, values, "1=1");
    }

    public int updateAllFeilds(String[] fieldNames, Object[] values) throws SmthDataAccessException {
        return updateFeilds(false, fieldNames, values, "1=1");
    }

    public int updateAllFeild(String fieldName, Object value) throws SmthDataAccessException {
        return updateFeild(fieldName, value, "1=1");
    }


//    public int updateFeild(Class<MODEL> entityClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
//        MODEL entity = null;
//        try {
//            entity = entityClass.newInstance();
//        } catch (Exception e) {
//            throw new AppletIllegalArgumentException(e.getMessage());
//        }
//        try {
//            BeanUtils.setProperty(entity, fieldName, value);
//        } catch (Exception e) {
//            throw new AppletIllegalArgumentException(e.getMessage());
//        }
//        return getJdbcDao().updateFeilds(entity, false, new String[]{fieldName}, where, params);
//    }

    public int updateAll(Collection<MODEL> entities) throws SmthDataAccessException {
        return getJdbcDao().updateAll(entities);
    }

    public int updateFeildsAll(Collection<MODEL> entities, String... fieldNames) throws SmthDataAccessException {
        return getJdbcDao().updateFeildsAll(entities, fieldNames);
    }
    //</editor-fold>

    //<editor-fold desc="delete">
    public boolean delete(MODEL entity) throws SmthDataAccessException {
        return getJdbcDao().delete(entity);
    }

    public int deleteAll(Collection<MODEL> entities) throws SmthDataAccessException {
        return getJdbcDao().deleteAll(entities);
    }

    public boolean deleteById(Serializable id) throws SmthDataAccessException {
        return getJdbcDao().deleteById(entityClass, id);
    }

    /**
     * 根据条件删除一组实体
     *
     * @param where d
     * @param params d
     * @return 返回信息
     */
    public int deleteModels(String where, Object... params) throws SmthDataAccessException {
        StringBuilder sb = new StringBuilder();
        sb.append(Table.getTable(entityClass).deleteClause());
        sb.append(" where ");
        sb.append(where);
        return getJdbcDao().update(sb.toString(), params);
    }

    public int deleteByIds(Collection<? extends KEY> ids) throws SmthDataAccessException {
        return getJdbcDao().deleteByIds(entityClass, ids);
    }
    //</editor-fold>

    //<editor-fold desc="findids">
    public List<KEY> findIds(String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().findIds(entityClass, where, params);
    }

    public List<KEY> findIds(int limit, boolean asceding) throws SmthDataAccessException {
        if (limit < 1) limit = 1;
        String sql = String.format("1=1 order by id %s limit %d", (asceding ? "asc" : "desc"), limit);
        return getJdbcDao().findIds(entityClass, sql);
    }
    //</editor-fold>

    public TailPage<MODEL> pageFindModels(WhereQuery query) {
        return pageFindModels(query.getSql(false), query.getPageNumber(), query.getPageSize(), query.sqlParams().toArray());
    }

    public List<MODEL> findModels(WhereQuery query) {
        return findModels(query.getSql(false), query.sqlParams().toArray());
    }


    public String getTableName() {
        return Table.getTable(entityClass).getTableName();
    }

    public List<MODEL> getFieldsByCollection(Collection beans, String fieldName) {
        return IdentifierKeyHelper.getFields(beans, fieldName);
    }

    public <T> void copyProperties(T orig, T dest) throws Exception {
        PropertyUtils.copyProperties(orig, dest);
    }


    //<editor-fold desc="getmodel">

    public MODEL getModel(boolean exclude, String[] fieldNames, String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().getModel(entityClass, exclude, fieldNames, where, params);
    }

    public MODEL getModel(String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().getModel(entityClass, where, params);
    }
    public MODEL getModel( boolean checkNull,boolean exclude, String[] fieldNames , String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull);
    }
    public MODEL getModel(boolean checkNull,String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull);
    }
    public MODEL getModel( boolean checkNull,String errorMessage,String errorCode,boolean exclude, String[] fieldNames , String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull, errorMessage, errorCode);
    }
    public MODEL getModel(boolean checkNull,String errorMessage,String errorCode,String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull, errorMessage, errorCode);
    }
    public MODEL getModel(boolean checkNull, SmthExceptionDictHandler exceptionEnum, boolean exclude, String[] fieldNames , String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(exclude, fieldNames, where, params), checkNull, exceptionEnum);
    }
    public MODEL getModel(boolean checkNull, SmthExceptionDictHandler exceptionEnum, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getModel(where, params), checkNull, exceptionEnum);
    }

    //</editor-fold>

    //<editor-fold desc="getcolumn">
    public <E> E getColumn(Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().getColumn(entityClass, columnClass, fieldName, where, params);
    }
    public <E> E getColumn(boolean checkNull,Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull);
    }
    public <E> E getColumn(boolean checkNull,String errorMessage,String errorCode,Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull, errorMessage, errorCode);
    }
    public <E> E getColumn(boolean checkNull, SmthExceptionDictHandler exceptionEnum, Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return SmthCheckExceptionUtil.checkIsNull(getColumn(columnClass, fieldName, where, params), checkNull, exceptionEnum);
    }


    //</editor-fold>

    //<editor-fold desc="count">
    public <T> int count(String where, Object... params) throws SmthDataAccessException {
        return getJdbcDao().count(entityClass, where, params);
    }
    //</editor-fold>


}
