package org.jsmth.jorm.service;

import org.jsmth.cache.Cache;
import org.jsmth.cache.CacheService;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.jorm.action.FindAllAction;
import org.jsmth.jorm.action.QueryAction;
import org.jsmth.domain.Identifier;
import org.jsmth.page.Page;
import org.jsmth.page.RollPage;
import org.jsmth.page.TailPage;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

/**
 * 提供实体缓存的基类Service
 * User: mason
 * Date: 2010-1-20
 * Time: 15:46:52
 */
@SuppressWarnings({"UnusedDeclaration"})
public abstract class EntityCacheableService<KEY extends Serializable, MODEL extends Identifier<KEY>, DAO extends EntityDao<KEY, MODEL>> extends EntityService<KEY, MODEL, DAO> {

    protected boolean useReadWriteEntityDao = false;
    protected boolean useCache = true;
    protected CacheService cacheService;
    protected int entityCacheMaxSize = 1000;
    protected int entityCacheMaxLiveSeconds = 3600;

    protected Cache entityCache;

    @PostConstruct
    @Override
    public void init() {
        super.init();
        Assert.notNull(cacheService, "cacheService must be set!");
        getEntityCache();
        useReadWriteEntityDao = (entityDao instanceof ReadWriteEntityDao);
    }

    protected Cache getEntityCache() {
        if (entityCache == null) {
            entityCache = cacheService.addCache(this.getClass().getName(), entityCacheMaxSize, entityCacheMaxLiveSeconds);
        }
        return entityCache;
    }

    @Override
    public MODEL getById(KEY id) {
        return getById(id, true);
    }

    public MODEL getById(KEY id, boolean useMasterExplicit) {
        if (useMasterExplicit)
            return EntityCacheHelper.getById(useCache, entityCacheMaxLiveSeconds, entityDao.entityClass, entityDao.getJdbcDao(), getEntityCache(), id);
        else
            return EntityCacheHelper.getById(useCache, entityCacheMaxLiveSeconds, entityDao.entityClass, ((ReadWriteEntityDao) entityDao).getSlaveJdbcDao(), getEntityCache(), id);
    }

    @Override
    public MODEL getOrCreateById(KEY id) {
        MODEL model = EntityCacheHelper.getById(useCache, entityCacheMaxLiveSeconds, entityDao.entityClass, entityDao.jdbcDao, getEntityCache(), id);
        if (model == null) {
            try {
                model = entityDao.getEntityClass().newInstance();
                model.setIdentifier(id);
                this.insert(model);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
        return model;
    }


    @Override
    public MODEL getModel(String where, Object... params) throws SmthDataAccessException {
        List<KEY> ids = findIds(where, params);
        List<MODEL> models = findByIds(ids);
        if(models==null || models.size()==0){
            return null;
        }
        return models.get(0);
    }

    @Override
    public List<MODEL> findByIds(List<KEY> ids) {
        return findByIds(ids, !useReadWriteEntityDao);
    }

    public List<MODEL> findByIds(List<KEY> ids, boolean useMasterExplicit) {
        if(ids==null || ids.size()==0){
            return new ArrayList<>();
        }
        if (useMasterExplicit)
            return EntityCacheHelper.findByIds(useCache, entityCacheMaxLiveSeconds, entityDao.entityClass, entityDao.jdbcDao, getEntityCache(), ids);
        else
            return EntityCacheHelper.findByIds(useCache, entityCacheMaxLiveSeconds, entityDao.entityClass, ((ReadWriteEntityDao) entityDao).getSlaveJdbcDao(), getEntityCache(), ids);
    }

    @Override
    public MODEL insert(MODEL entity) {
        return entityDao.insert(entity);
    }

    @Override
    public void insertAll(Collection<MODEL> entities) {
        entityDao.insertAll(entities);
    }

    @Override
    public int update(MODEL entity) {
        int ret = entityDao.update(entity);
        //实体缓存更新
        doUpdateEntityCache(entity);
        return ret;
    }


    @Override
    public int updateFields(MODEL entity, String... fieldNames) throws SmthDataAccessException {
        int ret = entityDao.updateFields(entity, fieldNames);
        //实体缓存更新
        doUpdateEntityCache(entity);
        return ret;
    }

    @Override
    public int updateAll(Collection<MODEL> entities) {
        int ret = entityDao.updateAll(entities);
        //实体缓存批量更新
        doUpdateEntityCache(entities);
        return ret;
    }


    @Override
    public int updateFeildsAll(Collection<MODEL> entities, String... fieldNames) throws SmthDataAccessException {
        int ret = entityDao.updateFeildsAll(entities, fieldNames);
        //实体缓存批量更新
        doUpdateEntityCache(entities);
        return ret;
    }


    @Override
    public MODEL save(MODEL entity) {
        int ret = this.update(entity);
        if (ret == 0) {
            this.insert(entity);
        }
        return entity;
    }

    @Override
    public boolean delete(MODEL entity) {
        boolean ret = entityDao.delete(entity);
        doDeleteEntityCache(entity);
        return ret;
    }

    @Override
    public int deleteAll(Collection<MODEL> entities) {
        int ret = entityDao.deleteAll(entities);
        doDeleteEntityCache(entities);
        return ret;
    }

    @Override
    protected int updateFeilds(boolean exclude, String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        int result = super.updateFeilds(exclude, fieldNames, values, where, params);
        doUpdateEntityCacheByWhere(where, params);
        return result;
    }

    @Override
    protected int updateFeilds(String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
        int result = super.updateFeilds(fieldNames, values, where, params);
        doUpdateEntityCacheByWhere(where, params);
        return result;
    }

    @Override
    protected int updateFeild(String fieldName, Object value, String where, Object... params) throws SmthDataAccessException {
        int result = super.updateFeild(fieldName, value, where, params);
        doUpdateEntityCacheByWhere(where, params);
        return result;
    }

    @Override
    protected int updateAllFeilds(boolean exclude, String[] fieldNames, Object[] values) throws SmthDataAccessException {
        int result = super.updateAllFeilds(exclude, fieldNames, values);
        doDeleteAllEntityCache();
        return result;
    }

    @Override
    protected int updateAllFeilds(String[] fieldNames, Object[] values) throws SmthDataAccessException {
        int result = super.updateAllFeilds(fieldNames, values);
        doDeleteAllEntityCache();
        return result;
    }

    @Override
    protected int updateAllFeild(String fieldName, Object value) throws SmthDataAccessException {
        int result = super.updateAllFeild(fieldName, value);
        doDeleteAllEntityCache();
        return result;
    }

    @Override
    public boolean deleteById(KEY id) throws SmthDataAccessException {
        boolean result = super.deleteById(id);
        doDeleteEntityCacheById(id);
        return result;
    }

    @Override
    public int deleteByIds(Collection<? extends KEY> ids) throws SmthDataAccessException {
        List<MODEL> models = findByIds(new LinkedList<KEY>(ids));
        int result = super.deleteByIds(ids);
        doDeleteEntityCache(models);
        return result;
    }

    @Override
    protected int deleteModels(String where, Object... params) throws SmthDataAccessException {
        List<MODEL> models = this.getEntityDao().findModels(where, params);
        int result = super.deleteModels(where, params);
        doUpdateEntityCache(models);
        return result;
    }

    // find all methods


    public Page<KEY> findAllIds(boolean asc, int pageNumber, int pageSize) {
        return new RollPage<KEY>(pageNumber, pageSize, new ArrayList<KEY>(getFindAllCacheAction(asc, pageNumber, pageSize).getIds()));
    }

    public Page<MODEL> findAll(boolean asc, int pageNumber, int pageSize) {
        return getFindAllCacheAction(asc, pageNumber, pageSize).getPage();
    }

    public TailPage<MODEL> findAllTail(boolean asc, int pageNumber, int pageSize) {
        return getFindAllCacheAction(asc, pageNumber, pageSize).getTailPage();
    }

    @SuppressWarnings({"unchecked"})
    protected FindAllAction<KEY, MODEL> getFindAllCacheAction(boolean asc, int pageNumber, int pageSize) {
        return (FindAllAction<KEY, MODEL>) buildQueryAction(FindAllAction.class, asc, pageNumber, pageSize);
    }

    public boolean isUseCache() {
        return useCache;
    }

    public CacheService getCacheService() {
        return cacheService;
    }

    public int getEntityCacheMaxSize() {
        return entityCacheMaxSize;
    }

    public int getEntityCacheMaxLiveSeconds() {
        return entityCacheMaxLiveSeconds;
    }

    public void setUseCache(boolean useCache) {
        this.useCache = useCache;
    }

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    public void setEntityCacheMaxSize(int entityCacheMaxSize) {
        this.entityCacheMaxSize = entityCacheMaxSize;
    }

    public void setEntityCacheMaxLiveSeconds(int entityCacheMaxLiveSeconds) {
        this.entityCacheMaxLiveSeconds = entityCacheMaxLiveSeconds;
    }


    protected void doDeleteAllEntityCache() {
        getEntityCache().clear();
    }

    protected void doDeleteEntityCache(MODEL entity) {
        EntityCacheHelper.onEntityDelete(getEntityCache(), entity);
    }
    protected void doDeleteEntityCacheById(KEY id) {
        EntityCacheHelper.onEntityIdsDelete(getEntityCache(), id);
    }

    protected void doDeleteEntityCache(Collection<MODEL> entities) {
        EntityCacheHelper.onEntityDelete(getEntityCache(), entities.toArray(new Identifier[entities.size()]));
    }

    protected void doDeleteEntityCacheByIds(Collection<KEY> ids) {
        List<KEY> keys = new LinkedList<KEY>(ids);
        List<MODEL> models = this.getEntityDao().findByIds(keys);
        doDeleteEntityCache(models);
    }

    protected void doDeleteEntityCacheByWhere(String where, Object... params) {
        List<MODEL> models = this.getEntityDao().findModels(where, params);
        doDeleteEntityCache(models);
    }

    protected void doUpdateEntityCache(MODEL entity) {
        if (useCache) {
            EntityCacheHelper.onEntityUpdate(entityCacheMaxLiveSeconds, getEntityCache(), entity);
        }
    }

    protected void doUpdateEntityCache(Collection<MODEL> entities) {
        if (useCache) {
            EntityCacheHelper.onEntityUpdate(entityCacheMaxLiveSeconds, getEntityCache(), entities.toArray(new Identifier[entities.size()]));
        }
    }

    protected void doUpdateEntityCacheByIds(Collection<KEY> ids) {
        List<KEY> keys = new LinkedList<KEY>(ids);
        List<MODEL> models = this.getEntityDao().findByIds(keys);
        doUpdateEntityCache(models);
    }

    protected void doUpdateEntityCacheByWhere(String where, Object... params) {
        List<MODEL> models = this.getEntityDao().findModels(where, params);
        doUpdateEntityCache(models);
    }

    protected <ACTION extends QueryAction<KEY, MODEL>> ACTION buildQueryAction(Class<ACTION> clazz, Object... params) {
        ACTION action;
        try {
            action = clazz.newInstance();
        } catch (Exception e) {
            throw new RuntimeException("CacheAction class[" + clazz + "] must be public and contains a default Constructor", e);
        }
        if (useReadWriteEntityDao) {
            action.prepare(getClass().getName(), isUseCache(), getEntityDao().getEntityClass(), ((ReadWriteEntityDao) getEntityDao()).getSlaveJdbcDao(), getEntityCache(), params);
        } else {
            action.prepare(getClass().getName(), isUseCache(), getEntityDao().getEntityClass(), getEntityDao().getJdbcDao(), getEntityCache(), params);
        }
        return action;
    }

    public boolean isUseReadWriteEntityDao() {
        return useReadWriteEntityDao;
    }

    public void setUseReadWriteEntityDao(boolean useReadWriteEntityDao) {
        this.useReadWriteEntityDao = useReadWriteEntityDao;
    }
}
