package org.jsmth.jorm.service;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.jsmth.cache.Cache;
import org.jsmth.util.IdentifierKeyHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;

import java.io.Serializable;
import java.util.*;

/**
 * @author mason
 */
@SuppressWarnings({"unchecked", "JavaDoc"})
public class EntityCacheHelper {
    protected static Logger log = LoggerFactory.getLogger(EntityCacheHelper.class);

    /**
     * 从缓存中按id获取数据，如果取不到则从数据库中取
     *
     * @param useCache d
     * @param entityClass d
     * @param jdbcDao d
     * @param entityCache d
     * @param id d
     * @param <KEY> d
     * @param <MODEL> d
     * @return 返回信息
     */
    public static <KEY extends Serializable, MODEL extends Identifier>  MODEL getById(boolean useCache, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, KEY id) {
        return getById(useCache, entityCache.getDefaultMaxLiveSeconds(), entityClass, jdbcDao, entityCache, id);
    }

    /**
     * 从缓存中按id获取数据 ，如果取不到则从数据库中取
     *
     * @param useCache d
     * @param maxLiveSeconds d
     * @param entityClass d
     * @param jdbcDao d
     * @param entityCache d
     * @param id d
     * @param <KEY> d
     * @param <MODEL> d
     * @return 返回信息
     */
    public static <KEY extends Serializable, MODEL extends Identifier>
    MODEL getById(boolean useCache, int maxLiveSeconds, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, KEY id) {
        return getById(false,useCache, maxLiveSeconds, entityClass, jdbcDao, entityCache, id);
    }
    public static <KEY extends Serializable, MODEL extends Identifier>
    MODEL getById(boolean checkNull,boolean useCache, int maxLiveSeconds, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, KEY id) {
        if (useCache) {
            String strid = String.valueOf(id);
            Object o = entityCache.get(strid);
            if (o != null) {
//                log.debug("get from cache "+String.valueOf( strid));
                return (MODEL) o;
            } else {
                MODEL byId = jdbcDao.getById(checkNull,entityClass, id);
//                log.debug("put to cache "+String.valueOf( strid));
                entityCache.put(strid, byId, maxLiveSeconds);
                return byId;
            }
        } else {
            return jdbcDao.getById(checkNull,entityClass, id);
        }
    }

    /**
     * 从缓存中按id列表获取一组数据，如果取不到则从数据库中取
     *
     * @param useCache d
     * @param entityClass d
     * @param jdbcDao d
     * @param entityCache d
     * @param ids d
     * @param <KEY> d
     * @param <MODEL> d
     * @return 返回信息
     */
    public static <KEY extends Serializable, MODEL extends Identifier>
    List<MODEL> findByIds(boolean useCache, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, List<KEY> ids) {
        return findByIds(useCache, entityCache.getDefaultMaxLiveSeconds(), entityClass, jdbcDao, entityCache, ids);
    }

    /**
     * 从缓存中按id列表获取一组数据，如果取不到则从数据库中取
     *
     * @param useCache d
     * @param maxLiveSeconds d
     * @param entityClass d
     * @param jdbcDao d
     * @param entityCache d
     * @param ids d
     * @param <KEY> d
     * @param <MODEL> d
     * @return 返回信息
     */
    public static <KEY extends Serializable, MODEL extends Identifier>
    List<MODEL> findByIds(boolean useCache, int maxLiveSeconds, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, List<KEY> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }

        if (useCache) {
            Map<String, Object> cacheModels = entityCache.getMulti(IdentifierKeyHelper.getStringList(ids));
            List<KEY> notCacheModelIds = new ArrayList<KEY>();

            for (KEY key : ids) {
                //从cache中取出来的都是toString的key
                Object o = cacheModels.get(key.toString());
                if (o == null) {
//                    System.out.printf("cache key " +entityClass.getName()+" "+key+" not find.");
                    notCacheModelIds.add(key);
//                    log.debug("get from cache "+String.valueOf( key));
                }
            }

            List<MODEL> addCacheModels = new ArrayList<MODEL>();
            //由于缓存失效，取不到的部分，一口气使用in查询从数据库获得
            List<MODEL> dbModels = jdbcDao.findByIds(entityClass, notCacheModelIds, true);
            List<MODEL> resultModels = new ArrayList<MODEL>();
            int count = 0;
            for (KEY key : ids) {
                //从cache中取出来的都是toString的key
                MODEL o = (MODEL) cacheModels.get(key.toString());
                if (o != null) {
//                    System.out.println("add cache object." + o);
                    resultModels.add(o);
                } else {
                    MODEL model = dbModels.get(count);
                    if (model != null) {
//                        System.out.println("add db object." + model);
                        resultModels.add(model);
                        addCacheModels.add(model);
//                        log.debug("put to cache "+String.valueOf( model.getIdentifier()));
                    }
                    count++;
                }
            }
            if (addCacheModels.size() > 0) {
                List<String> addCacheModelIds = IdentifierKeyHelper.getStringIdentifiers(addCacheModels);
                entityCache.putMulti(addCacheModelIds, addCacheModels, maxLiveSeconds);
            }
            return resultModels;
//            if (resultModels.size() == ids.size()) {
////                System.out.println("add objects to cache.");
//                //正常状况，将新查出来的部分缓存起来
//                List<String> tofillkeys = IdentifierKeyHelper.getStringIdentifiers(dbModels);
//                entityCache.putMulti(tofillkeys, dbModels, maxLiveSeconds);
//                return resultModels;
//
//            } else {
////                System.out.printf("select all update to cache.");
//                //异常情况，很有可能是数据库被修改而缓存未同步
//                List<MODEL> fresh = jdbcDao.findByIds(entityClass, ids, true);
//                List<String> freshkeys = IdentifierKeyHelper.getStringIdentifiers(resultModels);
//                entityCache.putMulti(freshkeys, fresh, maxLiveSeconds);
//                return fresh;
//            }
        } else {
            return jdbcDao.findByIds(entityClass, ids, true);
        }
    }

    public static <KEY extends Serializable, MODEL extends Identifier>
    List<MODEL> findByIds1(boolean useCache, int maxLiveSeconds, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, List<KEY> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Collections.emptyList();
        }

        if (useCache) {
            Map<String, Object> multi = entityCache.getMulti(IdentifierKeyHelper.getStringList(ids));
            List<KEY> nullids = new ArrayList<KEY>();

            for (KEY key : ids) {
                //从cache中取出来的都是toString的key
                Object o = multi.get(key.toString());
                if (o == null) {
//                    System.out.printf("cache key " +entityClass.getName()+" "+key+" not find.");
                    nullids.add(key);
                }
            }

            //由于缓存失效，取不到的部分，一口气使用in查询从数据库获得
            List<MODEL> tofill = jdbcDao.findByIds(entityClass, nullids, true);
            List<MODEL> ret = new ArrayList<MODEL>();
            int count = 0;
            for (KEY key : ids) {
                //从cache中取出来的都是toString的key
                MODEL o = (MODEL) multi.get(key.toString());
                if (o != null) {
//                    System.out.println("add cache object." + o);
                    ret.add(o);
                } else {
                    MODEL model = tofill.get(count);
                    if (model != null) {
//                        System.out.println("add db object." + model);
                        ret.add(model);
                    }
                    count++;
                }
            }
            if (ret.size() == ids.size()) {
//                System.out.println("add objects to cache.");
                //正常状况，将新查出来的部分缓存起来
                List<String> tofillkeys = IdentifierKeyHelper.getStringIdentifiers(tofill);
                entityCache.putMulti(tofillkeys, tofill, maxLiveSeconds);
                return ret;

            } else {
//                System.out.printf("select all update to cache.");
                //异常情况，很有可能是数据库被修改而缓存未同步
                List<MODEL> fresh = jdbcDao.findByIds(entityClass, ids, true);
                List<String> freshkeys = IdentifierKeyHelper.getStringIdentifiers(ret);
                entityCache.putMulti(freshkeys, fresh, maxLiveSeconds);
                return fresh;
            }
        } else {
            return jdbcDao.findByIds(entityClass, ids, true);
        }
    }

    public static <MODEL extends Identifier> void onEntityUpdate(Cache entityCache, MODEL... entities) {
        onEntityUpdate(entityCache.getDefaultMaxLiveSeconds(), entityCache, entities);
    }

    public static <MODEL extends Identifier> void onEntityUpdate(int maxLiveSeconds, Cache entityCache, MODEL... entities) {
        if (ArrayUtils.isEmpty(entities)) return;
        List<MODEL> list = Arrays.asList(entities);
        List<String> ids = IdentifierKeyHelper.getStringIdentifiers(list);
//        log.debug("put to caches ");
        entityCache.putMulti(ids, list, maxLiveSeconds);
    }

    public static <MODEL extends Identifier> void onEntityDelete(Cache entityCache, MODEL... entities) {
        if (ArrayUtils.isEmpty(entities)) return;
        //实体缓存删除
//        log.debug("remove from caches ");
        entityCache.removeMulti(IdentifierKeyHelper.getStringIdentifiers(Arrays.asList(entities)));
    }

    public static <KEY extends Serializable> void onEntityIdsDelete(Cache entityCache, Collection<KEY> keys) {
        if (CollectionUtils.isEmpty(keys)) return;
        Collection<String> tmp = new ArrayList<String>();
        for (KEY key : keys) {
            tmp.add(key.toString());
        }
//        log.debug("remove from caches ");
        entityCache.removeMulti(tmp);
    }

    public static <KEY extends Serializable> void onEntityIdsDelete(Cache entityCache, KEY... keys) {
        if (ArrayUtils.isEmpty(keys)) return;
        //实体缓存删除
        Collection<String> tmp = new ArrayList<String>();
        for (KEY key : keys) {
            tmp.add(key.toString());
        }
//        log.debug("remove from caches ");
        entityCache.removeMulti(tmp);
    }

    /**
     * 根据一组id从缓存中查询，如果查不到则在数据库中查
     *
     * @param maxLiveSeconds d
     * @param entityCache d
     * @param action dd
     * @param keys d
     * @return 返回信息
     */
    public static List findByIds(int maxLiveSeconds, Cache entityCache, DBAction action, List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return Collections.emptyList();
        }


        Map multi = entityCache.getMulti(keys);
        List<String> nullkeys = new ArrayList<String>();

        for (String key : keys) {
            Object o = multi.get(key);
            if (o == null) {
                nullkeys.add(key);
            }
        }

        //由于缓存失效，取不到的部分，一口气使用in查询从数据库获得
        List tofill = action.queryInDB(nullkeys);
        List ret = new ArrayList(keys.size());
        int count = 0;

        for (String key : keys) {
            Object o = multi.get(key);
            if (o != null) {
                ret.add(o);
            } else {
                ret.add(tofill.get(count));
                count++;
            }
        }

        if (ret.size() == keys.size()) {
            //正常状况，将新查出来的部分缓存起来
            List<String> tofillkeys = new ArrayList<String>();
            for (Object o : tofill) {
                if (o == null)
                    tofillkeys.add(null);
                else
                    tofillkeys.add(action.getKey(o));
            }
//            log.debug("put to caches ");
            entityCache.putMulti(tofillkeys, tofill, maxLiveSeconds);
            return ret;

        } else {
            //异常情况，很有可能是数据库被修改而缓存未同步
            List fresh = action.queryInDB(keys);
            List<String> freshkeys = new ArrayList<String>();
            for (Object o : fresh) {
                if (o == null)
                    freshkeys.add(null);
                else
                    freshkeys.add(action.getKey(o));
            }
//            log.debug("put to caches ");
            entityCache.putMulti(freshkeys, fresh, maxLiveSeconds);
            return fresh;
        }
    }

    public interface DBAction {

        List queryInDB(List<String> keys);

        String getKey(Object bean);
    }

}
