package org.jsmth.jorm.service;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.jorm.jdbc.BaseJdbcDao;
import org.jsmth.jorm.jdbc.EntityEventCallback;
import org.jsmth.jorm.jdbc.Event;
import org.jsmth.page.RollPage;
import org.jsmth.page.TailPage;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * User: 马生录（mason
 * Date: 12-2-9
 * Time: 下午10:14
 */
public class BaseEntityDao<MODEL> implements EntityEventCallback {

    protected Logger log = LoggerFactory.getLogger(this.getClass());

    protected EntityEventCallback entityEventCallback;

    protected BaseJdbcDao baseJdbcDao;

    protected Class<MODEL> entityClass;

    public BaseEntityDao(Class<MODEL> entityClass, EntityEventCallback entityEventCallback) {
        this.entityClass = entityClass;
        this.entityEventCallback = entityEventCallback;
    }

    public BaseEntityDao(Class<MODEL> entityClass) {
        this.entityClass = entityClass;
    }

    public BaseEntityDao(Class<MODEL> entityClass, BaseJdbcDao baseJdbcDao) {
        this.entityClass = entityClass;
        setBaseJdbcDao(baseJdbcDao);
    }

    public BaseJdbcDao getBaseJdbcDao() {
        return baseJdbcDao;
    }

    public void setBaseJdbcDao(BaseJdbcDao baseJdbcDao) {
        this.baseJdbcDao = baseJdbcDao;
        this.baseJdbcDao.setEntityEventCallback(this);
    }

    public Class<MODEL> getEntityClass() {
        return entityClass;
    }

    public void setEntityClass(Class<MODEL> entityClass) {
        this.entityClass = entityClass;
    }

    public List<MODEL> findAll() throws SmthDataAccessException {
        return getBaseJdbcDao().findAll(entityClass);
    }

    public TailPage<MODEL> findAll(int pageNumber, int pageSize) throws SmthDataAccessException {
        return getBaseJdbcDao().pageFindAll(entityClass, pageNumber, pageSize);
    }

    public RollPage<MODEL> RollFindAll(int pageNumber, int pageSize) throws SmthDataAccessException {
        return getBaseJdbcDao().rollFindAll(entityClass, pageNumber, pageSize);
    }

    /**
     * 根据条件，查询得到一组model
     *
     * @param where d
     * @param params d
     * @return 返回信息
     */
    public List<MODEL> findModels(String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().find(entityClass, where, params);
    }

    public TailPage<MODEL> pageFindModels(String where, int pageNumber, int pageSize, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().pageFind(entityClass, where, pageNumber, pageSize, params);
    }

    public RollPage<MODEL> rollFindModels(String where, int pageNumber, int pageSize, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().rollFind(entityClass, where, pageNumber, pageSize, params);
    }


    public <T, E> List<E> groupColumn(Class<E> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().groupColumn(entityClass, columnClass, fieldName, where, params);
    }

    public <T> List<T> findColumn(Class<T> columnClass, String fieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().findColumn(entityClass, columnClass, fieldName, where, params);
    }

    public <T> TailPage<T> pageFindColumn(Class<T> columnClass, String fieldName, String where, int pageNumber, int pageSize, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().pageFindColumn(entityClass, columnClass, fieldName, where, pageNumber, pageSize, params);
    }

    public <T> RollPage<T> rollFindColumn(Class<T> columnClass, String fieldName, String where, int pageNumber, int pageSize, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().rollFindColumn(entityClass, columnClass, fieldName, where, pageNumber, pageSize, params);
    }

    public <T, E> List<Map.Entry<E, Integer>> groupCountColumn(Class<E> columnClass, String fieldName, String countFieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().groupCountColumn(entityClass, columnClass, fieldName, countFieldName, where, params);
    }

    public <T, E> List<Map.Entry<E, Integer>> groupMaxColumn(Class<E> columnClass, String fieldName, String maxFieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().groupMaxColumn(entityClass, columnClass, fieldName, maxFieldName, where, params);
    }

    public <T, E> List<Map.Entry<E, Integer>> groupMinColumn(Class<E> columnClass, String fieldName, String minFieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().groupMinColumn(entityClass, columnClass, fieldName, minFieldName, where, params);
    }

    public <T, E> List<Map.Entry<E, Integer>> groupSumColumn(Class<E> columnClass, String fieldName, String sumFieldName, String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().groupSumColumn(entityClass, columnClass, fieldName, sumFieldName, where, params);
    }

    /**
     * 根据条件，查询得到记录总数
     *
     * @param where d
     * @param params d
     * @return 返回信息
     */
    public int countModels(String where, Object... params) throws SmthDataAccessException {
        return getBaseJdbcDao().count(entityClass, where, params);
    }

    /**
     * 根据条件，查询得到一个model。如果匹配的实体有多个，则返回第一个。
     *
     * @param where d
     * @param params d
     * @return 返回信息
     */
    public MODEL findUnique(String where, Object... params) throws SmthDataAccessException {
        List<MODEL> model = findModels(where, params);
        if (model.isEmpty()) {
            return null;
        } else {
            return model.get(0);
        }
    }

    @Override
    public void batchInvokeEvent(Collection targets, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.batchInvokeEvent(targets, event);
        }
    }

    @Override
    public void invokeEvent(Object target, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.invokeEvent(target, event);
        }
    }

    public EntityEventCallback getEntityEventCallback() {
        return entityEventCallback;
    }

    public void setEntityEventCallback(EntityEventCallback entityEventCallback) {
        this.entityEventCallback = entityEventCallback;
    }


}
