package org.jsmth.jorm.service;

import org.jsmth.cache.Cache;
import org.jsmth.cache.CacheService;
import org.jsmth.domain.Identifier;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import java.io.Serializable;

/**
 * 提供实体缓存的基类Service
 * User: mason
 * Date: 2010-1-20
 * Time: 15:46:52
 */
@SuppressWarnings({"UnusedDeclaration"})
public abstract class AbstractModelCacheableService<KEY extends Serializable, MODEL extends Identifier<KEY>> {
    protected boolean useCache = true;
    protected CacheService cacheService;
    protected int entityCacheMaxSize = 1000;
    protected int entityCacheMaxLiveSeconds = 3600;

    protected Cache entityCache;
    private Class<MODEL> modelClass;

    protected AbstractModelCacheableService(Class<MODEL> modelClass) {
        this.modelClass = modelClass;
    }

    @PostConstruct
    public void init() {
        Assert.notNull(cacheService, "cacheService must be set!");
        getEntityCache();
    }

    protected Cache getEntityCache() {
        if (entityCache == null) {
            entityCache = cacheService.addCache(this.getClass().getName(), entityCacheMaxSize, entityCacheMaxLiveSeconds);
        }
        return entityCache;
    }

    public MODEL getById(KEY id) {
        return (MODEL) entityCache.get(id);
    }

    public MODEL getOrCreateById(KEY id) {
        MODEL model = getById(id);
        if (model == null) {
            model = createModel(id);
            entityCache.put(id, model);
        }
        return model;
    }

    protected  MODEL createModel(KEY id){
        MODEL model = null;
        try {
            model = (MODEL)modelClass.newInstance();
            model.setIdentifier(id);
        } catch (InstantiationException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        return model;
    }


    public boolean insert(MODEL model) {
        String id = String.valueOf(model.getIdentifier());
        if (entityCache.containsKey(id)) {
//             System.out.println("online token " + model.getIdentifier() + " exist");
            return false;
        }
        return entityCache.put(id, model);
    }

    public boolean update(MODEL model) {
        String id = String.valueOf(model.getIdentifier());
        if (entityCache.containsKey(id)) {
//             System.out.println("online token " + model.getIdentifier() + " exist remove it");
            entityCache.remove(id);
        }
        return insert(model);
    }

    public void delete(KEY id) {
        String key = String.valueOf(id);
        if (entityCache.containsKey(key)) {
            System.out.println("online token " + id + " exist remove it");
            entityCache.remove(key);
        }
    }

    public void delete(MODEL entity) {
        if (entity == null) return;
        delete(entity.getIdentifier());
    }


    public CacheService getCacheService() {
        return cacheService;
    }

    public void setCacheService(CacheService cacheService) {
        this.cacheService = cacheService;
    }

    public int getEntityCacheMaxSize() {
        return entityCacheMaxSize;
    }

    public void setEntityCacheMaxSize(int entityCacheMaxSize) {
        this.entityCacheMaxSize = entityCacheMaxSize;
    }

    public int getEntityCacheMaxLiveSeconds() {
        return entityCacheMaxLiveSeconds;
    }

    public void setEntityCacheMaxLiveSeconds(int entityCacheMaxLiveSeconds) {
        this.entityCacheMaxLiveSeconds = entityCacheMaxLiveSeconds;
    }

    public boolean isUseCache() {
        return useCache;
    }

    public void setUseCache(boolean useCache) {
        this.useCache = useCache;
    }
}
