package org.jsmth.jorm.redis.dao;

import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.redis.jdbc.RedisDataSource;
import org.jsmth.jorm.redis.jdbc.RedisIdJdbcDao;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * MODEL: 马生录（mason
 * Date: 2015/3/12
 * Time: 21:04
 */
public class RedisEntityDao<KEY extends Serializable, MODEL extends Identifier<KEY>> {
    protected RedisDataSource redisDataSource;
    protected RedisIdJdbcDao<KEY, MODEL> jdbcDao;
    protected Class<MODEL> modelClass;
    protected Class<KEY> keyClass;
    String queueName = "QUEUE";
    String stackName = "STACK";


    public RedisEntityDao(Class<KEY> keyClass, Class<MODEL> modelClass) {
        this(keyClass, modelClass, null);
    }

    public RedisEntityDao(Class<KEY> keyClass, Class<MODEL> modelClass, RedisDataSource redisDataSource) {
        this.keyClass = keyClass;
        this.modelClass = modelClass;
        queueName = "QUEUE" + ":" + modelClass.getName();
        stackName = "STACK" + ":" + modelClass.getName();
        setRedisDataSource(redisDataSource);
    }

    public RedisDataSource getRedisDataSource() {
        return redisDataSource;
    }

    public void setRedisDataSource(RedisDataSource redisDataSource) {
        this.redisDataSource = redisDataSource;
        if (jdbcDao == null) {
            jdbcDao = new RedisIdJdbcDao<KEY, MODEL>(keyClass, modelClass);
        }
        jdbcDao.setRedisDataSource(redisDataSource, keyClass,keyClass, modelClass);
    }


    public RedisIdJdbcDao<KEY, MODEL> getJdbcDao() {
        return jdbcDao;
    }

    public void setJdbcDao(RedisIdJdbcDao<KEY, MODEL> jdbcDao) {
        this.jdbcDao = jdbcDao;
    }


    //<editor-fold desc="key value">
    /**
     * 新增
     * ------------------------------
     *
     * @param model d
     * @return 返回信息
     */
    public boolean add(final MODEL model) {
        boolean result = getJdbcDao().execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection)
                    throws SmthDataAccessException {
                RedisSerializer<KEY> keyserializer = (RedisSerializer<KEY>) getJdbcDao().getKeySerializer();
                byte[] key = keyserializer.serialize(model.getIdentifier());
                RedisSerializer<MODEL> valueserializer = (RedisSerializer<MODEL>) getJdbcDao().getValueSerializer();
                byte[] value = valueserializer.serialize(model);
                return connection.setNX(key, value);
            }
        });
        return result;
    }

    /**
     * 批量新增 使用pipeline方式
     * ------------------------------
     *
     * @param list d
     * @return 返回信息
     */
    public boolean add(final List<MODEL> list) {
        Assert.notEmpty(list);
        boolean result = getJdbcDao().execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection)
                    throws SmthDataAccessException {
                RedisSerializer<KEY> keyserializer = (RedisSerializer<KEY>) getJdbcDao().getKeySerializer();
                RedisSerializer<MODEL> valueserializer = (RedisSerializer<MODEL>) getJdbcDao().getValueSerializer();
                for (MODEL model : list) {
                    byte[] key = keyserializer.serialize(model.getIdentifier());
                    byte[] value = valueserializer.serialize(model);
                    connection.setNX(key, value);
                }
                return true;
            }
        }, false, true);
        return result;
    }

    /**
     * 删除
     * ------------------------------
     *
     * @param id d
     */
    public void delete(KEY id) {
        List<KEY> list = new ArrayList<KEY>();
        list.add(id);
        delete(list);
    }

    /**
     * 删除多个
     * ------------------------------
     *
     * @param ids d
     */
    public void delete(List<KEY> ids) {
        getJdbcDao().delete(ids);
    }

    /**
     * 修改
     * ------------------------------
     *
     * @param model d
     * @return 返回信息
     */
    public boolean update(final MODEL model) {
        KEY key = model.getIdentifier();
        if (get(key) == null) {
            throw new NullPointerException("data not exist, key = " + key);
        }
        boolean result = getJdbcDao().execute(new RedisCallback<Boolean>() {
            public Boolean doInRedis(RedisConnection connection)
                    throws SmthDataAccessException {
                RedisSerializer<KEY> keyserializer = (RedisSerializer<KEY>) getJdbcDao().getKeySerializer();
                byte[] key = keyserializer.serialize(model.getIdentifier());
                RedisSerializer<MODEL> valueserializer = (RedisSerializer<MODEL>) getJdbcDao().getValueSerializer();
                byte[] value = valueserializer.serialize(model);
                connection.set(key, value);
                return true;
            }
        });
        return result;
    }

    /**
     * 通过key获取
     * ------------------------------
     *
     * @param id d
     * @return 返回信息
     */
    public MODEL get(final KEY id) {
        MODEL result = getJdbcDao().execute(new RedisCallback<MODEL>() {
            public MODEL doInRedis(RedisConnection connection)
                    throws SmthDataAccessException {
                byte[] key = ((RedisSerializer<KEY>) getJdbcDao().getKeySerializer()).serialize(id);
                byte[] value = connection.get(key);
                if (value == null) {
                    return null;
                }
                MODEL model = ((RedisSerializer<MODEL>) getJdbcDao().getValueSerializer()).deserialize(value);
                return model;
            }
        });
        return result;
    }
    //</editor-fold>

    public boolean empty(){
        return true;//getJdbcDao().opsForList().size(getQueueName())
    }

    public String getQueueName() {
        return queueName;
    }

    public void setQueueName(String queueName) {
        this.queueName = queueName;
    }

    public String getStackName() {
        return stackName;
    }

    public void setStackName(String stackName) {
        this.stackName = stackName;
    }
}
