package org.jsmth.jorm.redis;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.BoundListOperations;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.CollectionUtils;

import java.lang.Exception;import java.lang.InterruptedException;import java.lang.Override;import java.lang.Short;import java.lang.String;import java.lang.Thread;import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * User: 马生录（mason
 * Date: 2015/3/12
 * Time: 20:57
 */
public class RedisQueue<T> implements InitializingBean, DisposableBean {
    private RedisTemplate redisTemplate;
    private String key;
    private int cap = Short.MAX_VALUE;//最大阻塞的容量，超过容量将会导致清空旧数据
    private byte[] rawKey;
    private RedisConnectionFactory factory;
    private RedisConnection connection;//for blocking
    private BoundListOperations<String, T> listOperations;//noblocking

    private Lock lock = new ReentrantLock();//基于底层IO阻塞考虑

    private QueueListenerCallback listener;//异步回调
    private Thread listenerThread;

    private boolean isClosed;

    public void setRedisTemplate(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    public void setListener(QueueListenerCallback listener) {
        this.listener = listener;
    }

    public void setKey(String key) {
        this.key = key;
    }


    @Override
    public void afterPropertiesSet() throws Exception {
        factory = redisTemplate.getConnectionFactory();
        connection = RedisConnectionUtils.getConnection(factory);
        rawKey = redisTemplate.getKeySerializer().serialize(key);
        listOperations = redisTemplate.boundListOps(key);
        if (listener != null) {
            listenerThread = new ListenerThread();
            listenerThread.setDaemon(true);
            listenerThread.start();
        }
    }


    /**
     * blocking
     * remove and get last item from queue:BRPOP
     * @param timeout d
     * @return 返回信息
     * @throws  InterruptedException d
     */
    public T takeFromTail(int timeout) throws InterruptedException {
        lock.lockInterruptibly();
        try {
            List<byte[]> results = connection.bRPop(timeout, rawKey);
            if (CollectionUtils.isEmpty(results)) {
                return null;
            }
            return (T) redisTemplate.getValueSerializer().deserialize(results.get(1));
        } finally {
            lock.unlock();
        }
    }

    public T takeFromTail() throws InterruptedException {
        return takeFromHead(0);
    }

    /**
     * 从队列的头，插入
     * @param value d
     */
    public void pushFromHead(T value) {
        listOperations.leftPush(value);
    }

    public void pushFromTail(T value) {
        listOperations.rightPush(value);
    }

    /**
     * noblocking
     *
     * @return 返回信息 null if no item in queue
     */
    public T removeFromHead() {
        return listOperations.leftPop();
    }

    public T removeFromTail() {
        return listOperations.rightPop();
    }

    /**
     * blocking
     * remove and get first item from queue:BLPOP
     * @param timeout d
     * @return 返回信息
     * @throws InterruptedException d
     */
    public T takeFromHead(int timeout) throws InterruptedException {
        lock.lockInterruptibly();
        try {
            List<byte[]> results = connection.bLPop(timeout, rawKey);
            if (CollectionUtils.isEmpty(results)) {
                return null;
            }
            return (T) redisTemplate.getValueSerializer().deserialize(results.get(1));
        } finally {
            lock.unlock();
        }
    }

    public T takeFromHead() throws InterruptedException {
        return takeFromHead(0);
    }

    @Override
    public void destroy() throws Exception {
        if (isClosed) {
            return;
        }
        shutdown();
        RedisConnectionUtils.releaseConnection(connection, factory);
    }

    private void shutdown() {
        try {
            listenerThread.interrupt();
        } catch (Exception e) {
            //
        }
    }

    class ListenerThread extends Thread {

        @Override
        public void run() {
            try {
                while (true) {
                    T value = takeFromHead();//cast exceptionyou should check.
                    //逐个执行
                    if (value != null) {
                        try {
                            listener.onMessage(value);
                        } catch (Exception e) {
                            //
                        }
                    }
                }
            } catch (InterruptedException e) {
                //
            }
        }
    }

    public static void main(String[] args) throws Exception {
        ClassPathXmlApplicationContext context = new ClassPathXmlApplicationContext("classpath:spring-redis-beans.xml");
        RedisQueue<String> redisQueue = (RedisQueue) context.getBean("jedisQueue");
        redisQueue.pushFromHead("test:app");
        Thread.sleep(15000);
        redisQueue.pushFromHead("test:app");
        Thread.sleep(15000);
        redisQueue.destroy();
    }

}
