package org.jsmth.jorm.jdbc;

import org.springframework.jdbc.IncorrectResultSetColumnCountException;
import org.springframework.jdbc.core.SingleColumnRowMapper;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.AbstractMap;
import java.util.Map;

/**
 * User: mason
 * Date: 2010-7-29
 * Time: 16:18:46
 */
public class TwoColumnRowMapper<K, V> extends SingleColumnRowMapper<Map.Entry<K, V>> {
    
    private Class<K> keyType;
    private Class<V> valueType;

    public TwoColumnRowMapper() {
    }

    public TwoColumnRowMapper(Class<K> keyType, Class<V> valueType) {
        this.keyType = keyType;
        this.valueType = valueType;
    }

    @SuppressWarnings({"unchecked"})
    @Override
    public Map.Entry<K, V> mapRow(ResultSet rs, int rowNum) throws SQLException {
        // Validate column count.
        ResultSetMetaData rsmd = rs.getMetaData();
        int nrOfColumns = rsmd.getColumnCount();
        if (nrOfColumns != 2) {
            throw new IncorrectResultSetColumnCountException(2, nrOfColumns);
        }

        // Extract column value from JDBC ResultSet.
        Object key = getColumnValue(rs, 1, keyType);
        Object value = getColumnValue(rs, 2, keyType);

        key = convertValueToRequiredTypeIfNecessary(key, keyType);
        value = convertValueToRequiredTypeIfNecessary(value, valueType);
        return new AbstractMap.SimpleEntry<K, V>((K) key, (V) value);
    }

    protected Object convertValueToRequiredTypeIfNecessary(Object value, Class requiredType) {
        Object ret;
        if (value != null && requiredType != null && requiredType.isInstance(value)) {
            ret = convertValueToRequiredType(value, requiredType);
        } else {
            ret = value;
        }
        return ret;
    }

}
