package org.jsmth.jorm.jdbc;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.Validate;
import org.jsmth.data.code.sqlbuilder.IllegalQueryException;
import org.jsmth.util.ReflectUtil;

import java.lang.reflect.*;
import java.util.*;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * 用以解析JPA annotation，支持常用的JPA annotation
 *
 * @author mason
 */
@SuppressWarnings({"JavaDoc", "UnusedDeclaration"})
public class Table<T> {

    private static final Map<Class, Table> tableCache = new HashMap<Class, Table>();

    @SuppressWarnings({"unchecked"})
    public static <T> Table<T> getTable(Class<T> clazz) {
        Table<T> ret = tableCache.get(clazz);
        if (ret == null) {
            synchronized (tableCache) {
                ret = tableCache.get(clazz);
                if (ret == null) {
                    ret = new Table<T>(clazz);
                }
                tableCache.put(clazz, ret);
            }
        }
        return ret;
    }

    Class<T> clazz;
    String tableName;
    Column idColumn;
    Class keyClass;


    Set<Column> columns = new LinkedHashSet<Column>();
    Map<String, Column> columnsByColumnName = new LinkedHashMap<String, Column>();
    Map<String, Column> columnsByFieldName = new LinkedHashMap<String, Column>();
    Map<Event, Method> events = new HashMap<Event, Method>(3);
    Map<String, Index> indexes = new LinkedHashMap<String, Index>();

//    Map<String, PropertyDescriptor> descriptors = new HashMap<String, PropertyDescriptor>();


    public Table(Class<T> clazz) {
        Validate.isTrue(JPAHelper.isEntity(clazz), "entity must annotated by @Entity");
        this.clazz = clazz;
        this.tableName = JPAHelper.getTableName(clazz);
        this.keyClass=(Class)ReflectUtil.getGenericClassParameterizedType(clazz);

        List<Field> fields = JPAHelper.getJPAFields(clazz);

        for (Field field : fields) {
            if (!JPAHelper.isEmbedded(field)) {
                Column col = new Column(field,keyClass);
                if (col.isIdentifier()) {
                    if (idColumn == null)
                        idColumn = col;
                    else
                        throw new IllegalArgumentException("Two or more ID field.");
                }
                columns.add(col);

            } else {
                List<Field> emfields = JPAHelper.getJPAFields(field.getType());
                for (Field emfield : emfields) {
                    if (!JPAHelper.isEmbedded(emfield)) {
                        Column col = new Column(emfield,keyClass);
                        col.setEmbedded(true);
                        //col.setEmbeddedField(field);
                        //将 fieldName 设置为  embeddedFieldName.innerFieldName
                        col.setFieldName(field.getName() + "." + emfield.getName());
//                        col.setFieldName(emfield.getName());
                        if (col.isIdentifier())
                            throw new IllegalArgumentException("embedded ID field.");
                        columns.add(col);
                    }
                }
            }
        }

        //子类属性重载部分
        Map<String, javax.persistence.Column> override = JPAHelper.getColumnOverride(clazz);

        for (Column column : columns) {
            String fieldName = column.getFieldName();
            javax.persistence.Column ann = override.get(fieldName);
            if (ann != null) {
                column.setColumnName(ann.name());
            }

            //数据表名可以全都小写
            columnsByColumnName.put(column.getColumnName().toLowerCase(), column);
            //属性名不能小写
            columnsByFieldName.put(column.getFieldName(), column);
        }

        //事件部分
        this.events.putAll(JPAHelper.getEventMap(clazz));

        //索引部分
        Set<Index> indexes = JPAHelper.getIndexFromClass(clazz);
        for (Index index : indexes) {
            this.indexes.put(index.getName(), index);
        }
        for (Field field : fields) {
            Index index = JPAHelper.getIndexFromField(field);
            if (index != null) {
                this.indexes.put(index.getName(), index);
            }
        }

        Validate.notNull(idColumn, "id column must not be null");
    }

    public void invokeEvent(Object target, Event event) {
        if (target == null) return;

        Method method = events.get(event);
        if (method != null) {
            try {
                method.invoke(target);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void batchInvokeEvent(Collection targets, Event event) {
        if (targets.isEmpty())
            return;

        if (!this.events.containsKey(event))
            return;

        Method method = events.get(event);

        for (Object target : targets) {
            if (target == null) continue;

            try {
                method.invoke(target);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * 生成形如 select * from AAA 的查询语句，用于便捷的拼接出复杂的查询语句
     *
     * @param exclude    true时使用全部field，而排除掉fieldNames中的部分；false时只使用fieldNames中的部分
     * @param fieldNames d
     * @return 返回信息
     */
    public String selectClause(boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        if (exclude) {
            Set<String> ommits = null;
            if (!ArrayUtils.isEmpty(fieldNames))
                ommits = new HashSet<String>(Arrays.asList(fieldNames));

            sql.append("select ");
            if (ommits == null) {
                sql.append(" * ");
            } else {
                for (Column col : columns) {
                    if (ommits.contains(col.getColumnName())) continue;
                    sql.append(col.getColumnName()).append(",");
                }
                sql.deleteCharAt(sql.length() - 1);
            }
        } else {
            Validate.notEmpty(fieldNames, "select clause should have one field");
            sql.append("select ");
            boolean found = false;
            for (String fieldName : fieldNames) {
                if (!columnsByFieldName.containsKey(fieldName)) continue;
                sql.append(fieldName).append(",");
                found = true;

            }
            //确实有合法的查询字段
            Validate.isTrue(found, "select clause should have one field");
            sql.deleteCharAt(sql.length() - 1);
        }
        sql.append(" from ").append(tableName);
        return sql.toString();
    }


    public List<Column> selectColumns(boolean excludeId, boolean exclude, String... fieldNames) {
        LinkedList<Column> items=new LinkedList<>();
        Set<String> fields=new LinkedHashSet<>();
        if(fieldNames!=null){
            for (String fieldName : fieldNames) {
                fields.add(fieldName);
            }
        }
        if (exclude) {
            for (Column column : columns) {
                if(column.isIdentifier() && excludeId){
                    continue;
                }
                if(fields.contains(column.fieldName)){
                    continue;
                }
                items.add(column);
            }
        } else {
            Validate.notEmpty(fieldNames, "select clause should have one field");
            for (Column column : columns) {
                if(column.isIdentifier() && excludeId){
                    continue;
                }
                if(fields.contains(column.fieldName)){
                    items.add(column);
                }
            }
        }
        return items;
    }


    public String distinctSelectClause(String fieldName) {
        StringBuilder sql = new StringBuilder();
        sql.append("select ");
        boolean found = false;
        //确实有合法的查询字段
        Validate.isTrue(columnsByFieldName.containsKey(fieldName), "select clause should have one field");
        sql.append("distinct ").append(fieldName);
        sql.append(" from ").append(tableName);
        return sql.toString();
    }

    public String selectGroupCountClause(String fieldName, String countFieldName) {
        StringBuilder sql = new StringBuilder();
        sql.append("select ");
        boolean found = false;
        //确实有合法的查询字段
        Validate.isTrue(columnsByFieldName.containsKey(fieldName), "select clause should have one field");
        sql.append("distinct ").append(fieldName);
        if (countFieldName == null || countFieldName.length() == 0)
            sql.append(",count(*)");
        else {
            Validate.isTrue(columnsByFieldName.containsKey(countFieldName), "select clause should have one field");
            sql.append(",count(").append(countFieldName).append(")");
        }
        sql.append(" from ").append(tableName);
        return sql.toString();
    }

    public String selectGroupSumClause(String fieldName) {
        return selectGroupSumClause(fieldName, "");
    }

    public String selectGroupSumClause(String fieldName, String sumFieldName) {
        StringBuilder sql = new StringBuilder();
        sql.append("select ");
        boolean found = false;
        //确实有合法的查询字段
        Validate.isTrue(columnsByFieldName.containsKey(fieldName), "select clause should have one field");
        sql.append("distinct ").append(fieldName);
        if (sumFieldName == null || sumFieldName.length() == 0)
            sql.append(",sum(*)");
        else {
            Validate.isTrue(columnsByFieldName.containsKey(sumFieldName), "select clause should have one field");
            sql.append(",sum(").append(sumFieldName).append(")");
        }
        sql.append(" from ").append(tableName);
        return sql.toString();
    }

    public String selectGroupMinClause(String fieldName, String minFieldName) {
        StringBuilder sql = new StringBuilder();
        sql.append("select ");
        boolean found = false;
        //确实有合法的查询字段
        Validate.isTrue(columnsByFieldName.containsKey(fieldName), "select clause should have one field");
        Validate.isTrue(columnsByFieldName.containsKey(minFieldName), "select clause should have one field");
        sql.append("distinct ").append(fieldName);
        sql.append(",min(").append(minFieldName).append(")");
        sql.append(" from ").append(tableName);
        return sql.toString();
    }

    public String selectGroupMaxClause(String fieldName, String maxFieldName) {
        StringBuilder sql = new StringBuilder();
        sql.append("select ");
        boolean found = false;
        //确实有合法的查询字段
        Validate.isTrue(columnsByFieldName.containsKey(fieldName), "select clause should have one field");
        Validate.isTrue(columnsByFieldName.containsKey(maxFieldName), "select clause should have one field");
        sql.append("distinct ").append(fieldName);
        sql.append(",max(").append(maxFieldName).append(")");
        sql.append(" from ").append(tableName);
        return sql.toString();
    }

    public String maxClause(String fieldName) {
        StringBuilder sql = new StringBuilder();
        Column col = columnsByFieldName.get(fieldName);
        Validate.notNull(col, String.format("no mapped column for fieldName [%s] of entity [%s] ", fieldName, clazz));
        sql.append("select max(").append(col.getColumnName()).append(")").append(" from ").append(tableName);
        return sql.toString();
    }

    public String minClause(String fieldName) {
        StringBuilder sql = new StringBuilder();
        Column col = columnsByFieldName.get(fieldName);
        Validate.notNull(col, String.format("no mapped column for fieldName [%s] of entity [%s] ", fieldName, clazz));
        sql.append("select min(").append(col.getColumnName()).append(")").append(" from ").append(tableName);
        return sql.toString();
    }

    public String countClause() {
        StringBuilder sql = new StringBuilder();
        sql.append("select count(*) from ").append(tableName);
        return sql.toString();
    }

    /**
     * 生成形如 select * from AAA where idColumn=? 的sql语句
     *
     * @param exclude d
     * @param fieldNames 指定不需要查询的类属性名称（注意不是字段名称）
     * @return 返回信息
     */
    public String selectById(boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        sql.append(selectClause(exclude, fieldNames)).append(" where ").append(byIdClause(false));
        return sql.toString();
    }

    /**
     * 生成形如 select * from AAA where idColumn in(?,?,?) 的sql语句
     * @param fieldNames d
     * @param exclude d
     * @param idCount d
     * @return 返回信息
     */
    public String selectByMultiId(int idCount, boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        sql.append(selectClause(exclude, fieldNames)).append(" where ").append(inIdsClause(idCount));
        return sql.toString();
    }

    /**
     * 生成形如 inser into AAA(a,b,c) values(:a,:b,:c)的namedQuery sql，如果是自增类型的，则不包含key
     *
     * @return 返回信息
     */
    public String insert() {
        return insert(false);
    }

    /**
     * 当生成形如 inser into AAA(a,b,c) values(:a,:b,:c)的namedQuery sql
     *
     * @param setPK true表示自增类型主键被强行设置了值
     * @return 返回信息
     */
    public String insert(boolean setPK) {
        StringBuilder sql = new StringBuilder();
        sql.append("insert into ").append(tableName).append("(");

        for (Column col : columns) {
            //只有没被设置过的自增id才不需要写入sql语句
            if (!setPK && idColumn.isIdAutoIncrease() && col.getColumnName().equals(idColumn.getColumnName()))
                continue;
            sql.append(col.getColumnName()).append(",");
        }
        sql.deleteCharAt(sql.length() - 1);
        sql.append(") values(");

        for (String s : columnsByFieldName.keySet()) {
            //只有没被设置过的自增id才不需要写入sql语句
            if (!setPK && idColumn.isIdAutoIncrease() && s.equals(idColumn.getFieldName()))
                continue;
            sql.append(":").append(s).append(",");
        }
        sql.deleteCharAt(sql.length() - 1);
        sql.append(")");
        return sql.toString();
    }

    /**
     * 生成形如 update AAA set x=:x,y=:y,z=:z1 where 1=1 的sql update片段
     *
     * @param exclude    true时使用全部field，而排除掉fieldNames中的部分；false时只使用fieldNames中的部分
     * @param fieldNames d
     * @return 返回信息
     */
    public String updateSetColumnClause(boolean exclude, String... fieldNames) {
        Set<String> ommits = null;
        if (!ArrayUtils.isEmpty(fieldNames))
            ommits = new HashSet<String>(Arrays.asList(fieldNames));

        StringBuilder sql = new StringBuilder();
        sql.append("update ").append(tableName).append(" set ");

        if (exclude) {
            for (Map.Entry<String, Column> entry : columnsByFieldName.entrySet()) {
                if (ommits != null && ommits.contains(entry.getKey())) continue;
                sql.append(entry.getValue().getColumnName()).append("=:").append(entry.getKey()).append(" ,");
            }
            sql.deleteCharAt(sql.length() - 1);
        } else {
            Validate.notEmpty(fieldNames, "update clause should have one field");

            boolean found = false;

            for (String fieldName : fieldNames) {
                Column col = columnsByFieldName.get(fieldName);
                if (col == null) continue;
                sql.append(col.getColumnName()).append("=:").append(fieldName).append(" ,");
                found = true;
            }
            sql.deleteCharAt(sql.length() - 1);

            Validate.isTrue(found, "update clause should have one field");

        }
        return sql.toString();
    }

    public String updateClause(boolean exclude, String... fieldNames) {
        Set<String> ommits = null;
        if (!ArrayUtils.isEmpty(fieldNames))
            ommits = new HashSet<String>(Arrays.asList(fieldNames));

        StringBuilder sql = new StringBuilder();
        sql.append("update ").append(tableName).append(" set ");

        if (exclude) {
            for (Map.Entry<String, Column> entry : columnsByFieldName.entrySet()) {
                if (ommits != null && ommits.contains(entry.getKey())) continue;
                sql.append(entry.getValue().getColumnName()).append("=?").append(" ,");
            }
            sql.deleteCharAt(sql.length() - 1);
        } else {
            Validate.notEmpty(fieldNames, "update clause should have one field");

            boolean found = false;

            for (String fieldName : fieldNames) {
                Column col = columnsByFieldName.get(fieldName);
                if (col == null) continue;
                sql.append(col.getColumnName()).append("=?").append(" ,");
                found = true;
            }
            sql.deleteCharAt(sql.length() - 1);

            Validate.isTrue(found, "update clause should have one field");

        }
        return sql.toString();
    }

//    public String updateDeltaClause(boolean exclude, List<String> fieldNames, List<String> deltaFieldNames) {
//        if (!exclude) {
//            Validate.notEmpty(fieldNames, "update clause should have one field");
//            //指定的部分，必须不拥有delta部分
//            Validate.isTrue(!CollectionUtils.containsAny(fieldNames, deltaFieldNames),"fieldNames & deltaFieldNames has intersection part.");
//        } else {
//            //排除掉的部分，必须完全拥有delta部分
//            Validate.isTrue(fieldNames.containsAll(deltaFieldNames), "fieldNames must contains all of deltaFieldNames");
//        }
//
//        StringBuilder sb = new StringBuilder(updateClause(exclude, fieldNames.toArray(new String[fieldNames.size()])));
//        for (String fieldName : deltaFieldNames) {
//            sb.append(", ")
//        }
//    }

    /**
     * 生成形如 update AAA set x=:x,y=:y,z=:z1 where idColumn=:idField 的sql语句
     *
     * @param exclude d
     * @param fieldNames d
     * @return 返回信息
     */
    public String updateById(boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        sql.append(updateSetColumnClause(exclude, fieldNames)).append(" where ").append(byIdClause(true));
        return sql.toString();
    }

    public String addFieldValueByWhere(String fieldName, String where) {
        StringBuilder sql = new StringBuilder();
        Column column = getColumnByFieldName(fieldName);
        sql.append("update ").append(tableName).append(" set ");
        sql.append(column.getColumnName()).append("=").append(column.getColumnName()).append("+?");
        if (where != null && where.length() > 0) {
            sql.append(" where ").append(where);
        }
        return sql.toString();
    }

    public String subFieldValueByWhere(String fieldName, String where) {
        StringBuilder sql = new StringBuilder();
        Column column = getColumnByFieldName(fieldName);
        sql.append("update ").append(tableName).append(" set ");
        sql.append(column.getColumnName()).append("=").append(column.getColumnName()).append("-?");
        if (where != null && where.length() > 0) {
            sql.append(" where ").append(where);
        }
        return sql.toString();
    }

    /**
     * 生成形如 update AAA set x=:x,y=:y,z=:z1 where idColumn=:idField 的sql语句
     *
     * @param exclude d
     * @param fieldNames d
     * @return 返回信息
     */
    public String updateSetFields(boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        sql.append(updateSetColumnClause(exclude, fieldNames));
        return sql.toString();
    }

    public String updateFields(boolean exclude, String... fieldNames) {
        StringBuilder sql = new StringBuilder();
        sql.append(updateClause(exclude, fieldNames));
        return sql.toString();
    }


    /**
     * 生成形如 delete from AAA where 1=1 的sql语句片段
     *
     * @return 返回信息
     */
    public String deleteClause() {
        StringBuilder sql = new StringBuilder();
        sql.append("delete from ").append(tableName);
        return sql.toString();
    }

    /**
     * 生成形如 delete from AAA where idColumn=? 的sql语句
     *
     * @return 返回信息
     */
    public String deleteById() {
        StringBuilder sql = new StringBuilder();
        sql.append(deleteClause()).append(" where ").append(byIdClause(false));
        return sql.toString();
    }

    /**
     * 生成形如 delete from AAA where idColumn in(?,?,?) 的sql语句
     *
     * @param idCount d
     * @return 返回信息
     */
    public String deleteByMultiId(int idCount) {
        StringBuilder sql = new StringBuilder();
        sql.append(deleteClause()).append(" where ").append(inIdsClause(idCount));
        return sql.toString();
    }


    public String byIdClause(boolean useNamedQuery) {
        StringBuilder sb = new StringBuilder();
        sb.append(idColumn.getColumnName());
        if (useNamedQuery)
            sb.append("=:").append(idColumn.getFieldName());
        else
            sb.append("=?");
        return sb.toString();
    }

    public String inIdsClause(int idCount) {
        return inClause(idColumn.getColumnName(), idCount);
    }

    public String inClause(String column, int count) {
        StringBuilder sb = new StringBuilder(column);
        sb.append(" in (");
        for (int i = 0; i < count; i++) {
            sb.append("?,");
        }
        sb.deleteCharAt(sb.length() - 1);
        sb.append(")");
        return sb.toString();
    }

    public Column getColumnByColumnName(String name) {
        return columnsByColumnName.get(name);
    }

    public Column getColumnByFieldName(String name) {
        return columnsByFieldName.get(name);
    }

//    public PropertyDescriptor getPropertyDescriptorByColumnName(String name) {
//        return descriptors.get(name);
//    }

    public List<Column> findColumns(boolean exclude,String... fieldNames){
        List<Column> columns=new ArrayList<>();
        Set<String> ommits = new HashSet<String>();
        if(fieldNames==null || fieldNames.length==0){
            if(!exclude){
                throw new IllegalQueryException("columnFields is zero or null.");
            }
        }else{
            ommits = new HashSet<String>(Arrays.asList(fieldNames));
        }

        if(exclude){
            for (Column column : this.columns) {
                if (ommits.contains(column.getFieldName())) continue;
                columns.add(column);
            }

        }else{
            for (Column column : this.columns) {
                if (ommits.contains(column.getFieldName())) {
                    columns.add(column);
                }
            }
        }
        return columns;
    }

    //gs

    public Class<T> getClazz() {
        return clazz;
    }

    public void setClazz(Class<T> clazz) {
        this.clazz = clazz;
    }

    public String getTableName() {
        return tableName;
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    public Column getIdColumn() {
        return idColumn;
    }

    public void setIdColumn(Column idColumn) {
        this.idColumn = idColumn;
    }

    public Set<Column> getColumns() {
        return columns;
    }

    public void setColumns(Set<Column> columns) {
        this.columns = columns;
    }

    public Map<String, Column> getColumnsByColumnName() {
        return columnsByColumnName;
    }

    public void setColumnsByColumnName(Map<String, Column> columnsByColumnName) {
        this.columnsByColumnName = columnsByColumnName;
    }

    public Map<String, Column> getColumnsByFieldName() {
        return columnsByFieldName;
    }

    public void setColumnsByFieldName(Map<String, Column> columnsByFieldName) {
        this.columnsByFieldName = columnsByFieldName;
    }

    public Map<Event, Method> getEvents() {
        return events;
    }

    public void setEvents(Map<Event, Method> events) {
        this.events = events;
    }

    public Map<String, Index> getIndexes() {
        return indexes;
    }

    public void setIndexes(Map<String, Index> indexes) {
        this.indexes = indexes;
    }

    //    public Map<String, PropertyDescriptor> getDescriptors() {
//        return descriptors;
//    }
//
//    public void setDescriptors(Map<String, PropertyDescriptor> descriptors) {
//        this.descriptors = descriptors;
//    }

    @Override
    public String toString() {
        return "Table{" +
                "clazz=" + clazz +
                ", tableName='" + tableName + '\'' +
//                ", idFieldType=" + idFieldType +
//                ", idFieldName='" + idFieldName + '\'' +
//                ", idColumnName='" + idColumnName + '\'' +
//                ", idAutoIncrease=" + idAutoIncrease +
//                ", fields=" + fields +
//                ", fieldAndColumn=" + fieldAndColumn +
//                ", columnAndField=" + columnAndField +
//                ", events=" + events +
                '}';
    }
}
