package org.jsmth.jorm.jdbc;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.cfg.AnnotationConfiguration;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.dialect.Dialect;
import org.hibernate.engine.spi.Mapping;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.Index;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.exception.SmthExceptionDict;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import javax.annotation.PostConstruct;
import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

/**
 * 在spring simpleJdbcTemplate的基础上扩展，满足基本CRUD业务的dao。
 * 在完全使用jdbc基础上，借助spring的声明式事务管理，获得最大的便利。
 *
 * @author mason
 */
@SuppressWarnings({"unchecked", "TypeParameterExplicitlyExtendsObject", "JavaDoc", "UnusedDeclaration"})
public class JdbcDao extends BaseTableJdbcDao {

    protected SchemaUpdateStrategy schemaUpdateStrategy = SchemaUpdateStrategy.UPDATE;

    public JdbcDao(DataSource dataSource) {
        super(dataSource);
    }

    public JdbcDao(DataSource dataSource, String dialect, String schemaUpdateStrategy) {
        super(dataSource);
        this.setDialect(dialect);
        this.setSchemaUpdateStrategy(schemaUpdateStrategy);
    }

    public void setSchemaUpdateStrategy(String strategy) {
        this.schemaUpdateStrategy = Enum.valueOf(SchemaUpdateStrategy.class, strategy);
        if (this.schemaUpdateStrategy == null)
            this.schemaUpdateStrategy = SchemaUpdateStrategy.NONE;
    }

    /**
     * 数据库更新策略
     *
     * @param schemaUpdateStrategy d
     */
    public void setSchemaUpdateStrategy(SchemaUpdateStrategy schemaUpdateStrategy) {
        this.schemaUpdateStrategy = schemaUpdateStrategy;
    }

    @PostConstruct
    public void init() throws SmthDataAccessException {
        Assert.notNull(dataSource, "dataSource must be set!");
        Assert.notNull(schemaUpdateStrategy, "schemaUpdateStrategy must be set!");
        Assert.notNull(hbdialect, "hbdialect must be set!");
    }

    public <T> void updateSchema(Class<T> entityClass) throws SmthDataAccessException {
        updateSchema(entityClass, schemaUpdateStrategy);
    }

    public <T> void updateSchema(Class<T> entityClass, boolean doCreateIndex) throws SmthDataAccessException {
        updateSchema(entityClass, schemaUpdateStrategy, doCreateIndex);
    }

    /**
     * 更新数据库schema，建表
     *
     * @param entityClass d
     * @param strategy d
     * @param <T> d
     * @throws SmthDataAccessException d
     */
    public <T> void updateSchema(Class<T> entityClass, SchemaUpdateStrategy strategy) throws SmthDataAccessException {
        updateSchema(entityClass, strategy, true);
    }

    public <T> void updateSchema(Class<T> entityClass, SchemaUpdateStrategy strategy, boolean doCreateIndex) throws SmthDataAccessException {
        if (strategy == SchemaUpdateStrategy.NONE)
            return;

        //认为表是存在的，这样比较好
        boolean tableExsist = true;

        if (strategy == SchemaUpdateStrategy.UPDATE) {
            try {
                executeDDL("desc " + JPAHelper.getTableName(entityClass));
            } catch (DataAccessException e) {
                tableExsist = false;
            }

            //表存在，就执行update操作，否则执行后面的create
            if (tableExsist) {
                //以前是执行 table 修补的sql，现在改为啥都不做了
            }
        }

        if (strategy == SchemaUpdateStrategy.CREATE || !tableExsist) {
            doCreateTable(entityClass, tableExsist);
            if (doCreateIndex)
                doCreateIndex(entityClass, tableExsist);
        }
    }

    public <T> boolean isTableExist(Class<T> entityClass) throws SmthDataAccessException {
        try {
            executeDDL("desc " + JPAHelper.getTableName(entityClass));
            return true;
        } catch (DataAccessException e) {
            return false;
        }
    }

    /**
     * 重建所有的表
     *
     * @param entityClass d
     * @throws SmthDataAccessException d
     */
    public void rebuildSchema(Class... entityClass) throws SmthDataAccessException {
        for (Class clazz : entityClass) {
            doCreateTable(clazz, true);
            doCreateIndex(clazz, true);
        }
    }

    /**
     * 删除所有的表的内容
     *
     * @param entityClass d
     * @throws SmthDataAccessException d
     */
    public void truncateSchema(Class... entityClass) throws SmthDataAccessException {
        for (Class clazz : entityClass) {
            doTruncateTable(clazz);
        }
    }

    /**
     * 删除表内容
     *
     * @param entityClass d
     * @param <T>  d
     * @throws SmthDataAccessException d
     */
    public <T> void doTruncateTable(Class<T> entityClass) throws SmthDataAccessException {
        AnnotationConfiguration cfg = new AnnotationConfiguration();
        cfg.addAnnotatedClass(entityClass);
        Mapping mapping = cfg.buildMapping();
        cfg.buildMappings();
        String defaultCatalog = cfg.getProperties().getProperty(Environment.DEFAULT_CATALOG);
        String defaultSchema = cfg.getProperties().getProperty(Environment.DEFAULT_SCHEMA);

        // hibernate table
        org.hibernate.mapping.Table table = getTable(cfg, hbdialect);
        //resort columns
        resortTableColumn(table, hbdialect, mapping);

        String sql;
        //默认情况下都执行，这个性能损耗很小
        sql = sqlTruncateString(table, hbdialect, defaultCatalog, defaultSchema);
        executeDDL(sql);
    }

    public String sqlTruncateString(org.hibernate.mapping.Table table,Dialect dialect, String defaultCatalog, String defaultSchema) {
        StringBuffer buf = new StringBuffer("truncate table ");
//        if(hbdialect.supportsIfExistsBeforeTableName()) {
//            buf.append("if exists ");
//        }
        buf.append(table.getQualifiedName(dialect, defaultCatalog, defaultSchema)).append(dialect.getCascadeConstraintsString());
//        if(hbdialect.supportsIfExistsAfterTableName()) {
//            buf.append(" if exists");
//        }
        return buf.toString();
    }

    /**
     * 创建表
     *
     * @param entityClass d
     * @param tableExsist 表已经存在，则先drop之
     * @param <T> d
     * @throws SmthDataAccessException d
     */
    public <T> void doCreateTable(Class<T> entityClass, boolean tableExsist) throws SmthDataAccessException {
        AnnotationConfiguration cfg = new AnnotationConfiguration();
        cfg.addAnnotatedClass(entityClass);
        Mapping mapping = cfg.buildMapping();
        cfg.buildMappings();
        String defaultCatalog = cfg.getProperties().getProperty(Environment.DEFAULT_CATALOG);
        String defaultSchema = cfg.getProperties().getProperty(Environment.DEFAULT_SCHEMA);

        // hibernate table
        org.hibernate.mapping.Table table = getTable(cfg, hbdialect);
        //resort columns
        resortTableColumn(table, hbdialect, mapping);

        String sql;
        //默认情况下都执行，这个性能损耗很小
        if (tableExsist) {
            sql = table.sqlDropString(hbdialect, defaultCatalog, defaultSchema);
            executeDDL(sql);
        }

        sql = table.sqlCreateString(hbdialect, mapping, defaultCatalog, defaultSchema);
        executeDDL(sql);
    }

    public <T> void doCreateIndex(Class<T> entityClass, boolean indexExsist) throws SmthDataAccessException {
        AnnotationConfiguration cfg = new AnnotationConfiguration();
        cfg.addAnnotatedClass(entityClass);
        Mapping mapping = cfg.buildMapping();
        cfg.buildMappings();
        String defaultCatalog = cfg.getProperties().getProperty(Environment.DEFAULT_CATALOG);
        String defaultSchema = cfg.getProperties().getProperty(Environment.DEFAULT_SCHEMA);

        // hibernate table
        org.hibernate.mapping.Table table = getTable(cfg, hbdialect);
        //resort columns
        resortTableColumn(table, hbdialect, mapping);

        //todo:  以后可以先检测一下，再删除索引，因为没有 drop index if exsits
//        if(indexExsist){
//            dropAllIndex(entityClass);
//        }

        Iterator indexes = table.getIndexIterator();
        String sql;
        while (indexes.hasNext()) {
            Index index = (Index) indexes.next();
            sql = index.sqlCreateString(hbdialect, mapping, defaultCatalog, defaultSchema);
            executeDDL(sql);
        }
    }


    /**
     * 更新表
     *
     * @param entityClass d
     * @param <T> d
     * @throws SmthDataAccessException d
     */
    @Deprecated
    public <T> void doUpdateTable(Class<T> entityClass) throws SmthDataAccessException {
        AnnotationConfiguration cfg = new AnnotationConfiguration();
        cfg.addAnnotatedClass(entityClass);
        Connection connection = null;
        String currentSql = "";
        try {
            connection = DataSourceUtils.getConnection(dataSource);
            String[] sqls;
            sqls = cfg.generateSchemaUpdateScript(hbdialect, new DatabaseMetadata(connection, hbdialect));
            for (String sql : sqls) {
                currentSql = sql;
                executeDDL(sql);
            }
        } catch (SQLException e) {
            DataSourceUtils.releaseConnection(connection, dataSource);
            connection = null;
            throw getExceptionTranslator().translate("ConnectionCallback", currentSql, e);
        } finally {
            DataSourceUtils.releaseConnection(connection, dataSource);
        }
    }

    protected org.hibernate.mapping.Table getTable(Configuration cfg, Dialect dialect) throws SmthDataAccessException {
        org.hibernate.mapping.Table table = (org.hibernate.mapping.Table) cfg.getTableMappings().next();
        Mapping mapping = cfg.buildMapping();
        resortTableColumn(table, dialect, mapping);
        table.sqlCreateString(dialect, mapping, cfg.getProperties().getProperty(Environment.DEFAULT_CATALOG), cfg.getProperties().getProperty(Environment.DEFAULT_SCHEMA));
        return table;
    }

    /**
     * 对字段进行重排序，hibernate只会根据字段默认次序排序，导致建表语句很烂
     *
     * @param table d
     * @param dialect d
     * @param mapping d
     * @throws SmthDataAccessException d
     */
    protected void resortTableColumn(org.hibernate.mapping.Table table, final Dialect dialect, final Mapping mapping) throws SmthDataAccessException {
        List<Column> columns = new LinkedList<Column>();
        List<Column> tempcolumns = new LinkedList<Column>();

        Set<Column> indexedColumn = new HashSet<Column>();
        Iterator idxit = table.getIndexIterator();
        while (idxit.hasNext()) {
            Index o = (Index) idxit.next();
            Iterator ci = o.getColumnIterator();
            while (ci.hasNext()) {
                Column column = (Column) ci.next();
                indexedColumn.add(column);
            }
        }
        indexedColumn.addAll(table.getPrimaryKey().getColumns());
        Iterator keyIterator = table.getForeignKeyIterator();
        while (keyIterator.hasNext()) {
            Column o = (Column) keyIterator.next();
            indexedColumn.add(o);
        }

        Iterator<Column> iterator = table.getColumnIterator();
        while (iterator.hasNext()) {
            Column column = iterator.next();
            //已经被索引的列，将直接按先后次序使用
            if (indexedColumn.contains(column)) {
                columns.add(column);
            } else {
                tempcolumns.add(column);
            }
        }

        //调整那些未被索引的列次序
        Collections.sort(tempcolumns, new Comparator<Column>() {
            @Override
            public int compare(Column o1, Column o2) {
                Integer order1 = JPAHelper.ORDERED_SQL_TYPES.get(o1.getSqlTypeCode(mapping));
                Integer order2 = JPAHelper.ORDERED_SQL_TYPES.get(o2.getSqlTypeCode(mapping));

                if (order1 != null && order2 != null) {
                    int ret = order1 - order2;
                    if(ret == 0){
                        ret = o1.getLength()  - o2.getLength();
                    }
                    return ret;
                } else if (order1 == null && order2 != null)
                    return -1;
                else if (order1 != null)
                    return 1;
                else {
                    //都是未知类型，则不排序
                    return 0;
                }
            }
        });

        columns.addAll(tempcolumns);

        Field field = ReflectionUtils.findField(org.hibernate.mapping.Table.class, "columns");
        ReflectionUtils.makeAccessible(field);
        try {
            Map map = (Map) field.get(table);
            map.clear();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        }
        for (Column column : columns) {
            table.addColumn(column);
        }
    }


    public <T> void dropTable(Class<T> entityClass) throws SmthDataAccessException {
        String sql = "drop table " + Table.getTable(entityClass).getTableName();
        executeDDL(sql);
    }

    public <T> void truncateTable(Class<T> entityClass) {
        String sql = "truncate table " + Table.getTable(entityClass).getTableName();
        executeDDL(sql);
    }

    public <T> boolean testIndexExist(Class<T> entityClass, String indexName) throws SmthDataAccessException {
        Table table = Table.getTable(entityClass);

        try {
            Connection conn = this.getDataSource().getConnection();
            DatabaseMetaData meta = conn.getMetaData();
            System.out.println(meta.getSchemaTerm());
            System.out.println(meta.getCatalogTerm());
            System.out.println("xxxxxxxxxxxxxxx");
            ResultSet result = meta.getIndexInfo(null, null, table.getTableName(), false, true);
            while (result.next()) {
                String name = result.getString(6);
                if (indexName.equals(name)) {
                    return true;
                }
            }
            return false;
        } catch (SQLException e) {
            throw new DataAccessException("", e) {
            };
        }
    }

    /**
     * 删除一个实体类对应表的所有索引
     *
     * @param entityClass d
     * @param <T> d
     * @throws SmthDataAccessException d
     */
    public <T> void dropAllIndex(Class<T> entityClass) throws SmthDataAccessException {
        Table<T> table = Table.getTable(entityClass);
        Set<String> names = table.getIndexes().keySet();
        dropIndex(entityClass, names.toArray(new String[names.size()]));
    }

    /**
     * 删除索引
     *
     * @param entityClass d
     * @param indexNames d
     * @param <T> d
     * @throws SmthDataAccessException d
     */
    public <T> void dropIndex(Class<T> entityClass, String... indexNames) throws SmthDataAccessException {
        if (ArrayUtils.isEmpty(indexNames))
            return;
        Table<T> table = Table.getTable(entityClass);
        for (String indexName : indexNames) {
            if (StringUtils.isBlank(indexName))
                continue;
            String sql = "alter table " + table.getTableName() + " drop index " + indexName;
            executeDDL(sql);
        }
    }

    public void executeDDL(String sql)  throws SmthDataAccessException {
        try {
            this.execute(sql);
        }catch (DataAccessException ex){
            throw new SmthDataAccessException(SmthExceptionDict.DataAccessException,ex);
        }
    }




}
