package org.jsmth.jorm.jdbc;

import org.apache.commons.lang.Validate;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.NotReadablePropertyException;
import org.springframework.beans.PropertyAccessorFactory;
import org.springframework.jdbc.core.StatementCreatorUtils;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

/**
 * 增强的SqlParameterSource，解决了枚举和Embedded的问题
 *
 * @author mason
 */
@SuppressWarnings({"unchecked"})
public class BeanPropertySqlParameterSourceEx implements SqlParameterSource {

    private Table table;

    private BeanWrapper beanWrapper;

    public BeanPropertySqlParameterSourceEx(Object object) {
        table = Table.getTable(object.getClass());
        this.beanWrapper = PropertyAccessorFactory.forBeanPropertyAccess(object);
    }


    public boolean hasValue(String paramName) {
        return true;
    }

    public Object getValue(String paramName) throws IllegalArgumentException {
        try {
            Column column = table.getColumnByFieldName(paramName);
            Validate.notNull(column, "no mapped field for " + paramName);

            if (column.isEnumerate()) {
                if (column.isUseOrdinal()) {
                    Enum value = (Enum) this.beanWrapper.getPropertyValue(column.getFieldName());
                    if (value != null)
                        return value.ordinal();
                    else
                        throw new IllegalArgumentException("Enum Field [" + column.getField() + "] can not be null while using Ordinal.");
                } else {
                    Object value = this.beanWrapper.getPropertyValue(column.getFieldName());
                    if (value != null)
                        return value.toString();
                    else
                        return null;
                }
            } else {
                return this.beanWrapper.getPropertyValue(column.getFieldName());
            }
        }
        catch (NotReadablePropertyException ex) {
            throw new IllegalArgumentException(ex.getMessage());
        }
    }

    /**
     * Derives a default SQL type from the corresponding property type.
     *
     * @see StatementCreatorUtils#javaTypeToSqlParameterType
     */
    public int getSqlType(String paramName) {
        Column column = table.getColumnByFieldName(paramName);

        Class propType = this.beanWrapper.getPropertyType(column.getFieldName());
        return StatementCreatorUtils.javaTypeToSqlParameterType(propType);
    }

    @Override
    public String getTypeName(String paramName) {
        return null;
    }

}
