package org.jsmth.jorm.iterator;

import org.apache.commons.lang.Validate;
import org.slf4j.LoggerFactory;
import org.jsmth.jorm.jdbc.ColumnType;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;
import org.jsmth.jorm.jdbc.Table;

import java.util.ArrayList;
import java.util.List;

/**
 * 适用于迭代获得long型自增id的表。
 * 注意，如果表记录数大于了 Integer.MAX_VALUE，则分页页号会出现问题，但迭代器本身的hasNext 和next()行为将被保证正常运转。
 * 
 * User: mason
 * Date: 2010-2-5
 * Time: 14:35:27
 */
public class LongIncreasedEntityBatchIterator<MODEL extends Identifier> extends EntityBatchIterator<Long, MODEL> {

    long maxId = -1;
    long currentId = 1l;
    boolean meetEnd;

    public LongIncreasedEntityBatchIterator(Class<MODEL> entityClazz, JdbcDao jdbcDao, int pageSize) {
        super(Long.class, entityClazz, jdbcDao, pageSize);
    }

    public LongIncreasedEntityBatchIterator(Class<MODEL> entityClazz, JdbcDao jdbcDao, long currentId, int pageSize) {
        super(Long.class, entityClazz, jdbcDao, pageSize);
        this.currentId = currentId;
    }

    public long getMaxId() {
        if (this.maxId == -1) {
            this.maxId = (int) jdbcDao.maxId(entityClazz, "1=1");
        }
        return this.maxId;
    }

    @Override
    protected void validKeyClass(Class<Long> keyClazz) {
    }

    @Override
    public boolean hasNext() {
        return !meetEnd;
    }

    @Override
    protected void validEntityClass(Class<MODEL> entityClazz) {
        Table<MODEL> table = Table.getTable(entityClazz);
        Validate.isTrue(table.getIdColumn().isIdAutoIncrease(), String.format("class [%s] must be auto increase entity", entityClazz));
        Validate.isTrue(table.getIdColumn().getType() == ColumnType.Long, String.format("class [%s] primary key must be long.", entityClazz));
    }

    @Override
    protected List<Long> getItemKeys(int pageNumber) {
        List<Long> ids = new ArrayList<Long>();

        long start = currentId;
        long end = currentId + getPageSize();

        if (end >= getMaxId()) {
            meetEnd = true;
        }

        for (long i = start; i < end; i++) {
            ids.add(i);
        }

        currentId = end;
        return ids;
    }

    @Override
    public int getTotalItemsCount() {
        if (getMaxId() > Integer.MAX_VALUE) {
            LoggerFactory.getLogger(LongIncreasedEntityBatchIterator.class).error("paging error because of the amount of class [%s] overflowed.", entityClazz);
            return Integer.MAX_VALUE;
        } else {
            return (int) getMaxId();
        }
    }
}
