package org.jsmth.jorm.iterator;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;
import org.jsmth.jorm.jdbc.Table;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 将id全部取出后，在内存中对全部id进行分页后，按页取对应的实体列表。
 * 此迭代器适用于任意情况的表，无论是否id自增都可以使用。
 * 并且，在id自增，但序列号断号情况比较严重的情况下，此方法性能更加。
 * 唯一需要注意的是，此方法不太适合于记录数极多的超级大表。
 * 经验数据有：
 * int型字段，5千万id全部加载内存大小为130m
 * string型字段，5千万id全部加载内存大小为420m
 * User: mason
 * Date: 2010-2-5
 * Time: 14:35:27
 */
public class KeyBasedEntityBatchIterator<KEY extends Serializable, MODEL extends Identifier<KEY>> extends EntityBatchIterator<KEY, MODEL> {

    protected int index = -1;
    protected KEY currentId;
    protected List<KEY> allIds;

    protected String where = "1=1 ";
    protected Object[] params;

    public KeyBasedEntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> clazz, JdbcDao jdbcDao, int pageSize) {
        super(keyClazz, clazz, jdbcDao, pageSize);
    }

    public KeyBasedEntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> clazz, JdbcDao jdbcDao, int pageSize, String where, Object... params) {
        super(keyClazz, clazz, jdbcDao, pageSize);
        this.where = where;
        this.params = params;

        Validate.notEmpty(where, "query string can not be empty, use 1=1 instead");
    }

    public KeyBasedEntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> clazz, JdbcDao jdbcDao, KEY currentId, int pageSize) {
        super(keyClazz, clazz, jdbcDao, pageSize);
        this.currentId = currentId;
        if (this.currentId != null) {
            this.where = Table.getTable(entityClazz).getIdColumn().getColumnName() + " >= " + currentId;
        }
    }

    public KeyBasedEntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> clazz, JdbcDao jdbcDao, KEY currentId, int pageSize, String where, Object... params) {
        super(keyClazz, clazz, jdbcDao, pageSize);
        this.currentId = currentId;

        if (currentId != null) {
            this.where = Table.getTable(entityClazz).getIdColumn().getColumnName() + " >= " + currentId;
        }
        if (!StringUtils.isBlank(where)) {
            this.where += where;
        }
        this.params = params;
    }

    public List<KEY> getAllIds() {
        if (this.allIds == null) {
            allIds = jdbcDao.findIds(entityClazz, where, params);
        }
        return this.allIds;
    }

    @Override
    public boolean hasNext() {
        return getIndex() < allIds.size();
    }

    @Override
    protected List<KEY> getItemKeys(int pageNumber) {
        List<KEY> ids = new ArrayList<KEY>();
        int start = getIndex();
        int end = start + getPageSize();

        if (start < 0)
            start = 0;

        if (end > getAllIds().size())
            end = getAllIds().size();

        for (int i = start; i < end; i++) {
            ids.add(allIds.get(i));
        }
        setIndex(end);
        return ids;
    }

    public int getIndex() {
        if (index == -1) {
            index = getAllIds().indexOf(currentId);
        }
        return index;
    }

    public void setIndex(int index) {
        this.index = index;
    }

    @Override
    public int getTotalItemsCount() {
        return getAllIds().size();
    }

    @Override
    protected void validKeyClass(Class<KEY> keyClazz) {
    }

    @Override
    protected void validEntityClass(Class<MODEL> entityClazz) {
    }

    @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
    private static void intMemoryUsage(int count) {
        //看看1千万的字符串id，能占多大的内存
        System.out.println(Runtime.getRuntime().totalMemory());
        List<Integer> ids = new ArrayList<Integer>(count);
        for (int i = 0; i < count; i++) {
            ids.add(count);
            if (i % 100000 == 0)
                System.out.println(Runtime.getRuntime().totalMemory());
        }
        System.out.println(Runtime.getRuntime().totalMemory());
    }

    @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
    private static void stringMemoryUsage(int count) {
        //看看1千万的字符串id，能占多大的内存
        System.out.println(Runtime.getRuntime().totalMemory());
        List<String> ids = new ArrayList<String>(count);
        for (int i = 0; i < count; i++) {
            ids.add(String.valueOf(count));
            if (i % 100000 == 0)
                System.out.println(Runtime.getRuntime().totalMemory());
        }
        System.out.println(Runtime.getRuntime().totalMemory());
    }


    public static void main(String[] args) {
        intMemoryUsage(50000000);
        System.gc();
        stringMemoryUsage(50000000);
        //50000000
        //int:133943296: 130m
        //string:420839424 :420m
    }
}
