package org.jsmth.jorm.iterator;

import org.apache.commons.lang.Validate;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

/**
 * 按分页的形式，批量的从数据库中迭代获取单表记录
 * 注意：这并不是一个线程安全的实现，也没有缓存考虑
 * User: mason
 * Date: 2010-2-5
 * Time: 15:27:40
 */
@SuppressWarnings({"JavaDoc"})
public abstract class EntityBatchIterator<KEY extends Serializable, MODEL extends Identifier> implements Iterator<List<MODEL>> {

    protected boolean ommitErrors = false;
    protected Class<KEY> keyClazz;
    protected Class<MODEL> entityClazz;
    protected JdbcDao jdbcDao;

    private int pageNumber = 1;
    private int pageSize = 1000;

    protected EntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> entityClazz, JdbcDao jdbcDao, int pageSize) {
        this(keyClazz, entityClazz, jdbcDao, 1, pageSize);
    }

    protected EntityBatchIterator(Class<KEY> keyClazz, Class<MODEL> entityClazz, JdbcDao jdbcDao, int pageNumber, int pageSize) {
        this.keyClazz = keyClazz;
        this.entityClazz = entityClazz;
        this.jdbcDao = jdbcDao;

        this.pageNumber = pageNumber;
        this.pageSize = pageSize;

        validKeyClass(keyClazz);
        validEntityClass(entityClazz);
        Validate.notNull(jdbcDao, "jdbcDao must be set");
        Validate.isTrue(pageSize > 0, "pageSize must larger than 0.");
    }

    protected abstract void validKeyClass(Class<KEY> keyClazz);

    protected abstract void validEntityClass(Class<MODEL> entityClazz);

    @Override
    public List<MODEL> next() {
        List<KEY> keys = getItemKeys(getPageNumber());
        List<MODEL> items = jdbcDao.findByIds(ommitErrors, entityClazz, keys, true);
        Iterator<MODEL> it = items.iterator();
        while (it.hasNext()) {
            MODEL model = it.next();
            if(model == null){
                it.remove();
            }
        }
        setPageNumber(getPageNumber() + 1);
        return items;
    }

    /**
     * 获得某一页对应所有key的列表
     *
     * @param pageNumber d
     * @return 返回信息
     */
    protected abstract List<KEY> getItemKeys(int pageNumber);

    @Override
    public void remove() {
        throw new UnsupportedOperationException();
    }

    //gs

    public int getPageNumber() {
        return pageNumber;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    public int getPageSize() {
        return pageSize;
    }

    public void setPageSize(int pageSize) {
        this.pageSize = pageSize;
    }

    public abstract int getTotalItemsCount();

    public int getLastPageNumber() {
        return (getTotalItemsCount() - 1) / getPageSize() + 1;
    }

    public boolean isOmmitErrors() {
        return ommitErrors;
    }

    public void setOmmitErrors(boolean ommitErrors) {
        this.ommitErrors = ommitErrors;
    }
}
