package org.jsmth.jorm.iterator;

import org.apache.commons.lang.Validate;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 将id全部取出后，在内存中对全部id进行分页后，按页取对应的实体列表。
 * 此迭代器适用于任意情况的表，无论是否id自增都可以使用。
 * 并且，在id自增，但序列号断号情况比较严重的情况下，此方法性能更加。
 * 唯一需要注意的是，此方法不太适合于记录数极多的超级大表。
 * 经验数据有：
 * int型字段，5千万id全部加载内存大小为130m
 * string型字段，5千万id全部加载内存大小为420m
 * User: mason
 * Date: 2010-2-5
 * Time: 14:35:27
 */
public class BothKeyEntityBatchIterator<ONEKEY extends Serializable, KEY extends Serializable, MODEL extends Identifier<KEY>> extends EntityBatchIterator<KEY, MODEL> {

    protected int index = -1;
    protected int currentIndex;
    protected List<ONEKEY> oneIds;
    protected Class<ONEKEY> oneKeyClazz;

    protected String where = "1=1 ";
    protected Object[] params;
    String keyOneName;

    public BothKeyEntityBatchIterator(Class<ONEKEY> oneKeyClazz, Class<KEY> keyClazz, Class<MODEL> entityClazz, JdbcDao jdbcDao, int pageSize, String keyOneName, String where) {
        super(keyClazz, entityClazz, jdbcDao, pageSize);
        this.oneKeyClazz = oneKeyClazz;
        this.keyOneName = keyOneName;
        this.where = where;
        Validate.notEmpty(where, "query string can not be empty, use 1=1 instead");
    }

    public BothKeyEntityBatchIterator(Class<ONEKEY> oneKeyClazz, Class<KEY> keyClazz, Class<MODEL> entityClazz, JdbcDao jdbcDao, int pageSize, String keyOneName, String where, Object[] params) {
        super(keyClazz, entityClazz, jdbcDao, pageSize);
        this.oneKeyClazz = oneKeyClazz;
        this.keyOneName = keyOneName;
        this.where = where;
        this.params = params;
        Validate.notEmpty(where, "query string can not be empty, use 1=1 instead");
    }


    public List<ONEKEY> getOneIds() {
        if (this.oneIds == null) {
            this.oneIds = jdbcDao.findDistinctColumn(entityClazz, oneKeyClazz, keyOneName, where, params);
            this.setPageNumber(oneIds.size());
            currentIndex = 0;
        }
        return this.oneIds;
    }

    @Override
    public boolean hasNext() {
        return getIndex() < oneIds.size();
    }

    @Override
    public List<MODEL> next() {
        int i = getIndex();
        if (getOneIds().size() == 0)
            return new ArrayList<MODEL>();
        ONEKEY onekey = getOneIds().get(i);

        Table<MODEL> table = Table.getTable(entityClazz);
        StringBuilder builder = new StringBuilder(where);
        builder.append(SQLHelper.whereAddColumn(keyOneName));
        List<Object> paras = new ArrayList<Object>();
        if (params != null) {
            for (Object param : params) {
                paras.add(param);
            }
        }
        paras.add(onekey);
        List<MODEL> items = jdbcDao.find(entityClazz, builder.toString(), paras.toArray());
        i++;
        setIndex(i);
        return items;
    }

    @Override
    protected List<KEY> getItemKeys(int pageNumber) {
        return new ArrayList<KEY>();
    }


    public int getIndex() {
        if (index == -1) {
            getOneIds();
            index = 0;
        }
        return index;
    }

    public void setIndex(int index) {
        this.index = index;
    }

    @Override
    public int getTotalItemsCount() {
        return getOneIds().size();
    }

    @Override
    protected void validKeyClass(Class<KEY> keyClazz) {
    }

    @Override
    protected void validEntityClass(Class<MODEL> entityClazz) {
    }

    @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
    private static void intMemoryUsage(int count) {
        //看看1千万的字符串id，能占多大的内存
        System.out.println(Runtime.getRuntime().totalMemory());
        List<Integer> ids = new ArrayList<Integer>(count);
        for (int i = 0; i < count; i++) {
            ids.add(count);
            if (i % 100000 == 0)
                System.out.println(Runtime.getRuntime().totalMemory());
        }
        System.out.println(Runtime.getRuntime().totalMemory());
    }

    @SuppressWarnings({"MismatchedQueryAndUpdateOfCollection"})
    private static void stringMemoryUsage(int count) {
        //看看1千万的字符串id，能占多大的内存
        System.out.println(Runtime.getRuntime().totalMemory());
        List<String> ids = new ArrayList<String>(count);
        for (int i = 0; i < count; i++) {
            ids.add(String.valueOf(count));
            if (i % 100000 == 0)
                System.out.println(Runtime.getRuntime().totalMemory());
        }
        System.out.println(Runtime.getRuntime().totalMemory());
    }


    public static void main(String[] args) {
        intMemoryUsage(50000000);
        System.gc();
        stringMemoryUsage(50000000);
        //50000000
        //int:133943296: 130m
        //string:420839424 :420m
    }
}
