package org.jsmth.jorm.domain.extension;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;

import javax.persistence.MappedSuperclass;
import javax.persistence.Transient;
import java.io.Serializable;
import java.util.*;

/**
 * 扩展属性类，提供了一个类似Map的接口，将所有属性存放在一个大map中，并使用xstream进行存储
 * 被embedded后，必须被代理的方法包括：getExtension_content,setExtension_content
 * 注意，不支持使用此抽象类嵌入模型中
 *
 * @author mason
 */
@SuppressWarnings({"unchecked"})
@MappedSuperclass
public abstract class Extension implements Serializable, Cloneable {
    public static final int MAPSTYLE_JSON = 1;
    public static final int MAPSTYLE_XML = 2;

    @Transient
    protected IExtensionModel extensionModel;

    @Transient
    protected int mapStyle = MAPSTYLE_JSON;

    /**
     * 在两个相同类型对象之间进行属性拷贝
     *
     * @param src d
     * @param desc  d
     * @param descriptorNames d
     * @return 返回信息 被修改的熟悉数目
     */
    public static int copy(Object src, Object desc, String... descriptorNames) {
        return copy(false, src, desc, descriptorNames);
    }

    /**
     * 在两个对象之间进行属性拷贝
     *
     * @param force           true则可以不要求两个对象的类型一致
     * @param src d
     * @param desc d
     * @param descriptorNames d
     * @return 返回信息 被修改的熟悉数目
     */
    public static int copy(boolean force, Object src, Object desc, String... descriptorNames) {
        if (ArrayUtils.isEmpty(descriptorNames))
            return 0;

        if (!force) {
            if (src.getClass() != desc.getClass()) {
                throw new IllegalArgumentException("src class[" + src.getClass() + "] is not same as desc class[" + desc.getClass() + "]");
            }
        }

        int changegd = 0;

        BeanWrapper o1 = new BeanWrapperImpl(src);
        BeanWrapper o2 = new BeanWrapperImpl(desc);

        Set<String> names = new HashSet<String>(Arrays.asList(descriptorNames));
        for (String name : names) {
            if (name == null)
                continue;

            Object value = o1.getPropertyValue(name);
            Object oldValue = o2.getPropertyValue(name);
            if (value != oldValue || value != null && value.equals(oldValue)) {
                o2.setPropertyValue(name, value);
                changegd++;
            }
        }
        return changegd;
    }

    public void copyFrom(Extension target) {
        this.setExtension_content(target.getExtension_content());
        this.map.putAll(target.map);
    }

    @Transient
    private Map<String, Object> map = new HashMap<String, Object>(0);


    protected abstract String mapToText(Map<String, Object> extension);

    protected abstract Map textToMap(String extension_content);

    public String getExtension_content(){
        return extensionModel.getExtension_Content();
    }

    public void setExtension_content(String s){
        extensionModel.setExtension_Content(s);
    }

    public void preInsert() {
        if (this.map.size() > 0) {
            this.setExtension_content(mapToText(map));
        }
    }

    public void preUpdate() {
        if (this.map.size() > 0) {
            this.setExtension_content(mapToText(map));
        }
    }

    public void postLoad() {
        String x = getExtension_content();
        if (!StringUtils.isBlank(x))
            this.map = textToMap(x);
    }

    //<editor-fold desc="Prop">

    /**
     * 获得属性值
     *
     * @param <T> d
     * @param key d
     * @param defaultVal    如果key对应的值不存在，则返回此默认值
     * @param putIfNotExsit 如果不存在，则用默认值进行填充
     * @return 返回信息
     */
    public <T> T get(String key, T defaultVal, boolean putIfNotExsit) {
        if (StringUtils.isBlank(key) || defaultVal == null && putIfNotExsit) return null;

        T o = (T) this.map.get(key);
        if (o == null) {
            o = defaultVal;
            if (putIfNotExsit) {
                this.put(key, o);
            }
        }
        return o;
    }


    /**
     * 获得属性值
     *
     * @param <T> d
     * @param key d
     * @param defaultVal d
     * @return 返回信息
     */
    public <T> T get(String key, T defaultVal) {
        return get(key, defaultVal, false);
    }

    /**
     * 获得属性值
     *
     * @param key d
     * @return 返回信息
     */
    public Object get(String key) {
        if (StringUtils.isBlank(key)) return null;
        return this.map.get(key);
    }

    /**
     * 获得String类型的属性值
     *
     * @param key d
     * @return 返回信息
     */
    public String getStrProp(String key) {
        if (StringUtils.isBlank(key)) return null;

        Object prop = this.get(key);
        if (prop instanceof String) {
            return (String) prop;
        } else {
            return null;
        }
    }

    public Set<String> keySet() {
        return this.map.keySet();
    }

    public boolean containsKey(String key) {
        return this.map.containsKey(key);
    }

    public boolean containsValue(Object val) {
        return this.map.containsValue(val);
    }

    public Object remove(String key) {
        return this.map.remove(key);
    }

    public Object put(String key, Object val, boolean overrided) {
        if (containsKey(key)) {
            if (overrided)
                remove(key);
        }
        return put(key, val);
    }

    public Object coverPut(String key, Object val) {
        if (StringUtils.isBlank(key) || val == null) return null;
        if (containsKey(key)) {
            this.map.remove(key);
        }
        return put(key, val);
    }

    public Object put(String key, Object val) {
        if (StringUtils.isBlank(key) || val == null) return null;
        return this.map.put(key, val);
    }
    //</editor-fold>

    //<editor-fold desc="list">

    public <T> List<T> getList(Class<T> classz, String key) {
        return getList(classz, key, new ArrayList<T>());
    }

    public <T> List<T> getList(Class<T> classz, String key, Object defaultValue) {
        return (List<T>) get(key, defaultValue);
    }

    public void putList(String key, List values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="ListItem">
    public <T> boolean containsListItem(Class<T> classz, String key, T value) {
        List<T> list = getList(classz, key);
        return list.contains(value);
    }


    public <T> void removeListItem(Class<T> classz, String key, int index) {
        List<T> list = getList(classz, key);
        list.remove(index);
        putList(key, list);
    }

    public <T> void removeListItem(Class<T> classz, String key, Object value) {
        List<T> list = getList(classz, key);
        list.remove(value);
        putList(key, list);
    }

    public <T> T getListItem(Class<T> classz, String key, int index) {
        List<T> list = getList(classz, key);
        if (index < list.size())
            return list.get(index);
        else
            return null;
    }

    public <T> int putListItem(Class<T> classz, String key, T value) {
        return putListItem(classz, key, value, false);
    }

    public <T> int putListItem(Class<T> classz, String key, T value, boolean override) {
        List<T> list = getList(classz, key);

        if (list.contains(value)) {
            if (override) {
                list.remove(value);
            }
        }
        list.add(value);
        putList(key, list);
        return list.size();
    }

    //</editor-fold>

    //<editor-fold desc="ValueIntList">
    public List<Integer> getValueIntList(String key) {
        return getValueIntList(key, new ArrayList<Integer>());
    }

    public List<Integer> getValueIntList(String key, List<Integer> defaultValues) {
        return getList(Integer.class, key, defaultValues);
    }

    public int getValueIntListItem(String key, int index) {
        return getListItem(Integer.class, key, index);
    }

    public void removeValueIntListItem(String key, int index) {
        removeListItem(Integer.class, key, index);
    }

    public void removeValueIntListItem(String key, Object value) {
        removeListItem(Integer.class, key, value);
    }

    public int putValueIntListItem(String key, int value, boolean override) {
        return putListItem(Integer.class, key, value, override);
    }

    public int putValueIntListItem(String key, int value) {
        return putListItem(Integer.class, key, value);
    }

    //</editor-fold>
    //<editor-fold desc="ValueStringList">
    public List<String> getValueStringList(String key) {
        return getValueStringList(key, new ArrayList<String>());
    }

    public List<String> getValueStringList(String key, List<String> defaultValues) {
        return getList(String.class, key, defaultValues);
    }

    public String getValueStringListItem(String key, int index) {
        return getListItem(String.class, key, index);
    }

    public void removeValueStringListItem(String key, int index) {
        removeListItem(String.class, key, index);
    }

    public void removeValueStringListItem(String key, Object value) {
        removeListItem(String.class, key, value);
    }

    public int putValueStringListItem(String key, String value, boolean override) {
        return putListItem(String.class, key, value, override);
    }

    public int putValueStringListItem(String key, String value) {
        return putListItem(String.class, key, value);
    }

    //</editor-fold>

    //<editor-fold desc="map">
    public <K, V> Map<K, V> getMap(Class<K> classKey, Class<V> classValue, String key) {
        return getMap(classKey, classValue, key, new LinkedHashMap<K, V>());
    }

    public <K, V> Map<K, V> getMap(Class<K> classKey, Class<V> classValue, String key, Object defaultValue) {
        return (Map<K, V>) get(key, defaultValue);
    }

    public void putMap(String key, Map values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="mapItem">
    public <K, V> V getMapItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, V> map = getMap(classKey, classValue, key);
        if (map.containsKey(itemKey))
            return map.get(itemKey);
        else
            return null;
    }

    public <K, V> int putMapItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, V value) {
        return putMapItem(classKey, classValue, key, itemKey, value, false);
    }

    public <K, V> int putMapItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, V value, boolean override) {
        Map<K, V> map = getMap(classKey, classValue, key);
        if (map.containsKey(itemKey)) {
            if (override) {
                map.remove(itemKey);
            }
        }
        map.put(itemKey, value);
        putMap(key, map);
        return map.size();
    }

    public <K, V> boolean containsMapItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, V> map = getMap(classKey, classValue, key);
        return map.containsKey(itemKey);
    }


    public <K, V> void removeMapItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, V> map = getMap(classKey, classValue, key);
        map.remove(itemKey);
        putMap(key, map);
    }

    //</editor-fold>

    //<editor-fold desc="KeyStringMap">
    public <V> boolean containsKeyStringMapItem(Class<V> classValue, String key, String itemKey) {
        Map<String, V> map = getKeyStringMap(classValue, key);
        return map.containsKey(itemKey);
    }

    public <V> Map<String, V> getKeyStringMap(Class<V> classValue, String key) {
        return getMap(String.class, classValue, key, new LinkedHashMap());
    }

    public <V> Map<String, V> getKeyStringMap(Class<V> classValue, String key, Map<String, V> defaultValues) {
        return getMap(String.class, classValue, key, defaultValues);
    }

    public <V> V getKeyStringMapItem(Class<V> classValue, String key, String itemKey) {
        return getMapItem(String.class, classValue, key, itemKey);
    }

    public <V> void removeKeyStringMapItem(Class<V> classValue, String key, String itemKey) {
        removeMapItem(String.class, classValue, key, itemKey);
    }

    public <V> int putKeyStringMapItem(Class<V> classValue, String key, String itemKey, V value, boolean override) {
        return putMapItem(String.class, classValue, key, itemKey, value, override);
    }

    public <V> int putKeyStringMapItem(Class<V> classValue, String key, String itemKey, V value) {
        return putMapItem(String.class, classValue, key, itemKey, value);
    }

    //</editor-fold>
    //<editor-fold desc="KeyIntMap">
    public <V> boolean containsKeyIntMapItem(Class<V> classValue, String key, Integer itemKey) {
        Map<Integer, V> map = getKeyIntMap(classValue, key);
        return map.containsKey(itemKey);
    }

    public <V> Map<Integer, V> getKeyIntMap(Class<V> classValue, String key) {
        return getMap(Integer.class, classValue, key, new LinkedHashMap());
    }

    public <V> Map<Integer, V> getKeyIntMap(Class<V> classValue, String key, Map<String, V> defaultValues) {
        return getMap(Integer.class, classValue, key, defaultValues);
    }

    public <V> V getKeyIntMapItem(Class<V> classValue, String key, Integer itemKey) {
        return getMapItem(Integer.class, classValue, key, itemKey);
    }

    public <V> void removeKeyIntMapItem(Class<V> classValue, String key, Integer itemKey) {
        removeMapItem(Integer.class, classValue, key, itemKey);
    }

    public <V> int putKeyIntMapItem(Class<V> classValue, String key, Integer itemKey, V value, boolean override) {
        return putMapItem(Integer.class, classValue, key, itemKey, value, override);
    }

    public <V> int putKeyIntMapItem(Class<V> classValue, String key, Integer itemKey, V value) {
        return putMapItem(Integer.class, classValue, key, itemKey, value);
    }

    //</editor-fold>
    //<editor-fold desc="ValueStringMap">
    public Map<String, String> getValueStringMap(String key) {
        return getKeyStringMap(String.class, key, new LinkedHashMap());
    }

    public Map<String, String> getValueStringMap(String key, Map<String, String> defaultValues) {
        return getKeyStringMap(String.class, key, defaultValues);
    }

    public String getValueStringMapItem(String key, String itemKey) {
        return getKeyStringMapItem(String.class, key, itemKey);
    }

    public void removeValueStringMapItem(String key, String itemKey) {
        removeKeyStringMapItem(String.class, key, itemKey);
    }

    public int putValueStringMapItem(String key, String itemKey, String value, boolean override) {
        return putKeyStringMapItem(String.class, key, itemKey, value, override);
    }

    public int putValueStringMapItem(String key, String itemKey, String value) {
        return putKeyStringMapItem(String.class, key, itemKey, value);
    }

    //</editor-fold>
    //<editor-fold desc="ValueIntMap">
    public Map<String, Integer> getValueIntMap(String key) {
        return getKeyStringMap(Integer.class, key, new LinkedHashMap());
    }

    public Map<String, Integer> getValueIntMap(String key, Map<String, Integer> defaultValues) {
        return getKeyStringMap(Integer.class, key, defaultValues);
    }

    public Integer getValueIntMapItem(String key, String itemKey) {
        return getKeyStringMapItem(Integer.class, key, itemKey);
    }

    public void removeValueIntMapItem(String key, String itemKey) {
        removeKeyStringMapItem(Integer.class, key, itemKey);
    }

    public int putValueIntMapItem(String key, String itemKey, Integer value, boolean override) {
        return putKeyStringMapItem(Integer.class, key, itemKey, value, override);
    }

    public int putValueIntMapItem(String key, String itemKey, Integer value) {
        return putKeyStringMapItem(Integer.class, key, itemKey, value);
    }
    //</editor-fold>

    //<editor-fold desc="MapMap">
    public <K, SK, V> Map<K, Map<SK, V>> getMapMap(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key) {
        return getMapMap(classKey, classSubKey, classValue, key, new LinkedHashMap<K, Map<SK, V>>());
    }

    public <K, SK, V> Map<K, Map<SK, V>> getMapMap(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, Object defaultValue) {
        return (Map<K, Map<SK, V>>) get(key, defaultValue);
    }

    public void putMapMap(String key, Map values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="MapMapItem">
    public <K, SK, V> Map<SK, V> getMapMapItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey) {
        Map<K, Map<SK, V>> map = getMapMap(classKey, classSubKey, classValue, key);
        if (map.containsKey(itemKey))
            return map.get(itemKey);
        else
            return new LinkedHashMap<SK, V>();
    }

    public <K, SK, V> int putMapMapItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, Map<SK, V> value) {
        return putMapMapItem(classKey, classSubKey, classValue, key, itemKey, value, false);
    }

    public <K, SK, V> int putMapMapItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, Map<SK, V> value, boolean override) {
        Map<K, Map<SK, V>> map = getMapMap(classKey, classSubKey, classValue, key);
        if (map.containsKey(itemKey)) {
            if (override) {
                map.remove(itemKey);
            }
        }
        map.put(itemKey, value);
        putMap(key, map);
        return map.size();
    }

    public <K, SK, V> boolean containsMapMapItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey) {
        Map<K, Map<SK, V>> map = getMapMap(classKey, classSubKey, classValue, key);
        return map.containsKey(itemKey);
    }

    public <K, SK, V> void removeMapMapItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey) {
        Map<K, Map<SK, V>> map = getMapMap(classKey, classSubKey, classValue, key);
        map.remove(itemKey);
        putMap(key, map);
    }

    //</editor-fold>
    //<editor-fold desc="MapMapItemItem">
    public <K, SK, V> V getMapMapItemItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, SK subItemKey) {
        Map<SK, V> map = getMapMapItem(classKey, classSubKey, classValue, key, itemKey);
        if (map.containsKey(subItemKey))
            return map.get(subItemKey);
        else
            return null;
    }

    public <K, SK, V> int putMapMapItemItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, SK subItemKey, V value) {
        return putMapMapItemItem(classKey, classSubKey, classValue, key, itemKey, subItemKey, value, false);
    }

    public <K, SK, V> int putMapMapItemItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, SK subItemKey, V value, boolean override) {
        Map<SK, V> map = getMapMapItem(classKey, classSubKey, classValue, key, itemKey);
        if (map.containsKey(subItemKey)) {
            if (override) {
                map.remove(subItemKey);
            }
        }
        map.put(subItemKey, value);
        putMapMapItem(classKey, classSubKey, classValue, key, itemKey, map);
        return map.size();
    }

    public <K, SK, V> boolean containsMapMapItemItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, SK subItemKey) {
        Map<SK, V> map = getMapMapItem(classKey, classSubKey, classValue, key, itemKey);
        return map.containsKey(subItemKey);
    }

    public <K, SK, V> void removeMapMapItemItem(Class<K> classKey, Class<SK> classSubKey, Class<V> classValue, String key, K itemKey, SK subItemKey) {
        Map<SK, V> map = getMapMapItem(classKey, classSubKey, classValue, key, itemKey);
        map.remove(subItemKey);
        putMapMapItem(classKey, classSubKey, classValue, key, itemKey, map);
    }

    //</editor-fold>

    //<editor-fold desc="MapList">
    public <K, V> Map<K, List<V>> getMapList(Class<K> classKey, Class<V> classValue, String key) {
        return getMapList(classKey, classValue, key, new LinkedHashMap<K, List<V>>());
    }

    public <K, V> Map<K, List<V>> getMapList(Class<K> classKey, Class<V> classValue, String key, Object defaultValue) {
        return (Map<K, List<V>>) get(key, defaultValue);
    }

    public void putMapList(String key, Map values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="MapListItem">
    public <K, V> List<V> getMapListItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, List<V>> map = getMapList(classKey, classValue, key);
        if (map.containsKey(itemKey))
            return map.get(itemKey);
        else
            return new ArrayList<V>();
    }

    public <K, V> int putMapListItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, List<V> value) {
        return putMapListItem(classKey, classValue, key, itemKey, value, false);
    }

    public <K, V> int putMapListItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, List<V> value, boolean override) {
        Map<K, List<V>> map = getMapList(classKey, classValue, key);
        if (map.containsKey(itemKey)) {
            if (override) {
                map.remove(itemKey);
            }
        }
        map.put(itemKey, value);
        putMap(key, map);
        return map.size();
    }

    public <K, V> boolean containsMapListItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, List<V>> map = getMapList(classKey, classValue, key);
        return map.containsKey(itemKey);
    }

    public <K, V> void removeMapListItem(Class<K> classKey, Class<V> classValue, String key, K itemKey) {
        Map<K, List<V>> map = getMapList(classKey, classValue, key);
        map.remove(itemKey);
        putMap(key, map);
    }

    //</editor-fold>
    //<editor-fold desc="MapListItemItem">
    public <K, V> V getMapListItemItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, int index) {
        List<V> list = getMapListItem(classKey, classValue, key, itemKey);
        if (index < list.size())
            return list.get(index);
        else
            return null;
    }

    public <K, V> int putMapListItemItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, V value) {
        return putMapListItemItem(classKey, classValue, key, itemKey, value, false);
    }

    public <K, V> int putMapListItemItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, V value, boolean override) {
        List<V> list = getMapListItem(classKey, classValue, key, itemKey);
        if (list.contains(value)) {
            if (override) {
                list.remove(value);
            }
        }
        list.add(value);
        putMapListItem(classKey, classValue, key, itemKey, list);
        return list.size();
    }

    public <K, V> boolean containsMapListItemItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, V value) {
        List<V> list = getMapListItem(classKey, classValue, key, itemKey);
        return list.contains(value);
    }

    public <K, V> void removeMapListItemItem(Class<K> classKey, Class<V> classValue, String key, K itemKey, int index) {
        List<V> list = getMapListItem(classKey, classValue, key, itemKey);
        list.remove(index);
        putMapListItem(classKey, classValue, key, itemKey, list);
    }

    //</editor-fold>

    //<editor-fold desc="ListList">
    public <V> List<List<V>> getListList(Class<V> classValue, String key) {
        return getListList(classValue, key, new ArrayList<List<V>>());
    }

    public <V> List<List<V>> getListList(Class<V> classValue, String key, Object defaultValue) {
        return (List<List<V>>) get(key, defaultValue);
    }

    public void putListList(String key, List values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="ListListItem">
    public <V> List<V> getListListItem(Class<V> classValue, String key, int index) {
        List<List<V>> list = getListList(classValue, key);
        if (index < list.size())
            return list.get(index);
        else
            return null;
    }

    public <V> int putListListItem(Class<V> classValue, String key, int index, List<V> value) {
        return putListListItem(classValue, key, index, value, false);
    }

    public <V> int putListListItem(Class<V> classValue, String key, int index, List<V> value, boolean override) {
        List<List<V>> list = getListList(classValue, key);
        if (list.size() > index) {
            if (override) {
                list.remove(index);
            }
        }
        list.add(index, value);
        putListList(key, list);
        return list.size();
    }

    public <V> boolean containsListListItem(Class<V> classValue, String key, int index, List<V> value) {
        List<List<V>> list = getListList(classValue, key);
        return list.contains(value);
    }

    public <V> void removeListListItem(Class<V> classValue, String key, int index) {
        List<List<V>> list = getListList(classValue, key);
        list.remove(index);
        putListList(key, list);
    }

    //</editor-fold>
    //<editor-fold desc="ListListItemItem">
    public <V> V getListListItemItem(Class<V> classValue, String key, int index, int subIndex) {
        List<V> list = getListListItem(classValue, key, index);
        if (index < list.size())
            return list.get(subIndex);
        else
            return null;
    }

    public <V> int putListListItemItem(Class<V> classValue, String key, int index, int subIndex, V value) {
        return putListListItemItem(classValue, key, index, subIndex, value, false);
    }

    public <V> int putListListItemItem(Class<V> classValue, String key, int index, int subIndex, V value, boolean override) {
        List<V> list = getListListItem(classValue, key, index);
        if (list.size() > subIndex) {
            if (override) {
                list.remove(subIndex);
            }
        }
        list.add(subIndex, value);
        putListListItem(classValue, key, index, list);
        return list.size();
    }

    public <V> boolean containsListListItemItem(Class<V> classValue, String key, int index, V value) {
        List<V> list = getListListItem(classValue, key, index);
        return list.contains(value);
    }

    public <V> void removeListListItemItem(Class<V> classValue, String key, int index, int subIndex) {
        List<V> list = getListListItem(classValue, key, index);
        list.remove(subIndex);
        putListListItem(classValue, key, index, list);
    }

    //</editor-fold>

    //<editor-fold desc="ListMap">
    public <SK, V> List<Map<SK, V>> getListMap(Class<SK> classSubKey, Class<V> classValue, String key) {
        return getListMap(classSubKey, classValue, key, new ArrayList<Map<SK, V>>());
    }

    public <SK, V> List<Map<SK, V>> getListMap(Class<SK> classSubKey, Class<V> classValue, String key, Object defaultValue) {
        return (List<Map<SK, V>>) get(key, defaultValue);
    }

    public void putListMap(String key, List values) {
        put(key, values, true);
    }

    //</editor-fold>
    //<editor-fold desc="ListMapItem">
    public <SK, V> Map<SK, V> getListMapItem(Class<SK> classSubKey, Class<V> classValue, String key, int index) {
        List<Map<SK, V>> list = getListMap(classSubKey, classValue, key);
        if (index < list.size())
            return list.get(index);
        else
            return null;
    }

    public <SK, V> int putListMapItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, Map<SK, V> value) {
        return putListMapItem(classSubKey, classValue, key, index, value, false);
    }

    public <SK, V> int putListMapItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, Map<SK, V> value, boolean override) {
        List<Map<SK, V>> list = getListMap(classSubKey, classValue, key);
        if (list.size() > index) {
            if (override) {
                list.remove(index);
            }
        }
        list.add(index, value);
        putListMap(key, list);
        return list.size();
    }

    public <SK, V> boolean containsListMapItem(Class<SK> classSubKey, Class<V> classValue, String key, Map<SK, V> value) {
        List<Map<SK, V>> list = getListMap(classSubKey, classValue, key);
        return list.contains(value);
    }

    public <SK, V> void removeListMapItem(Class<SK> classSubKey, Class<V> classValue, String key, int index) {
        List<Map<SK, V>> list = getListMap(classSubKey, classValue, key);
        list.remove(index);
        putListMap(key, list);
    }

    //</editor-fold>
    //<editor-fold desc="ListMapItemItem">
    public <SK, V> V getListMapItemItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, SK subItemKey) {
        Map<SK, V> map = getListMapItem(classSubKey, classValue, key, index);
        if (map.containsKey(subItemKey))
            return map.get(subItemKey);
        else
            return null;
    }

    public <SK, V> int putListMapItemItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, SK subItemKey, V value) {
        return putListMapItemItem(classSubKey, classValue, key, index, subItemKey, value, false);
    }

    public <SK, V> int putListMapItemItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, SK subItemKey, V value, boolean override) {
        Map<SK, V> map = getListMapItem(classSubKey, classValue, key, index);
        if (map.containsKey(subItemKey)) {
            if (override) {
                map.remove(subItemKey);
            }
        }
        map.put(subItemKey, value);
        putListMapItem(classSubKey, classValue, key, index, map);
        return map.size();
    }

    public <SK, V> boolean containsListMapItemItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, SK subItemKey) {
        Map<SK, V> map = getListMapItem(classSubKey, classValue, key, index);
        return map.containsKey(subItemKey);
    }

    public <SK, V> void removeListMapItemItem(Class<SK> classSubKey, Class<V> classValue, String key, int index, SK subItemKey) {
        Map<SK, V> map = getListMapItem(classSubKey, classValue, key, index);
        map.remove(subItemKey);
        putListMapItem(classSubKey, classValue, key, index, map);
    }

//</editor-fold>

    public int getMapStyle() {
        return mapStyle;
    }

    public void setMapStyle(int mapStyle) {
        this.mapStyle = mapStyle;
    }
}
