package org.jsmth.jorm.action;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.Validate;
import org.jsmth.jorm.service.EntityCacheHelper;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.JdbcDao;
import org.jsmth.cache.Cache;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * 查询action。
 * 带有实体缓存，但不进行查询缓存的操作
 * User: mason
 * Date: 2010-1-20
 * Time: 14:52:54
 */
@SuppressWarnings({"JavaDoc"})
public abstract class QueryAction<KEY extends Serializable, MODEL extends Identifier> {

    protected String name;
    protected boolean useCache;
    protected Class<MODEL> entityClass;
    protected JdbcDao jdbcDao;

    protected Cache entityCache;
    protected StringBuilder sql = new StringBuilder();
    protected List sqlParams = new ArrayList();
    protected Object[] params;

    public QueryAction() {
    }

    /**
     * 准备查询action所需的参数
     * @param name s
     * @param useCache s
     * @param entityClass s
     * @param jdbcDao s
     * @param entityCache s
     * @param params s
     */
    public void prepare(String name, boolean useCache, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache, Object... params){
        prepareBasic(name, useCache, entityClass, jdbcDao, entityCache);
        prepareParam(params);
        prepareSQL();
    }

    protected void prepareBasic(String name, boolean useCache, Class<MODEL> entityClass, JdbcDao jdbcDao, Cache entityCache) {
        this.name = name;
        this.useCache = useCache;

        Validate.notNull(name, "action name must be set!");
        Validate.notNull(entityClass, "entityClass must be set!");
        Validate.notNull(entityCache, "entityCache must be set!");
        Validate.notNull(jdbcDao, "jdbcDao must be set!");
        this.entityClass = entityClass;
        this.entityCache = entityCache;
        this.jdbcDao = jdbcDao;
    }

    protected void prepareParam(Object... params){
        this.params = params;
    }

    /**
     * 返回本查询的model结果列表
     *
     * @return 返回信息
     */
    public List<MODEL> getModels() {
        List<KEY> ids = new ArrayList<KEY>(getIds());
        return EntityCacheHelper.findByIds(useCache, entityClass, jdbcDao, entityCache, ids);
    }

    /**
     * 获得本查询的id结果列表
     *
     * @return 返回信息
     */
    public List<KEY> getIds() {
        return getIdsInDB();
    }

    /**
     * 数据库中根据sql，查询得到
     * @return 返回信息
     */
    protected List<KEY> getIdsInDB() {
        if (CollectionUtils.isEmpty(sqlParams))
            return jdbcDao.findIds(entityClass, sql.toString());
        else
            return jdbcDao.findIds(entityClass, sql.toString(), sqlParams.toArray());
    }

    /**
     * 供子类重载，根据action构造时传入的多参数params，构造出sql和sqlParams
     */
    abstract protected void prepareSQL();


}
