/*
Copyright (c) 2008 Health Market Science, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.jsmth.data.sqlbuilder;

import com.healthmarketscience.common.util.AppendableExt;

import java.io.IOException;
import java.util.Date;

/**
 * Outputs SQL and a prefix enclosed within JDBC escape syntax
 * "{&lt;prefix&gt; &lt;sql&gt;}".
 *
 * @author James Ahlborn
 */
public class JdbcEscape extends Expression
{

  /**
   * Enum which defines the escape types supported.
   */
  public enum Type
  {
    SCALAR_FUNCTION("fn "),
    DATE("d "),
    TIME("t "),
    TIMESTAMP("ts "),
    OUTER_JOIN("oj "),
    STORED_PROCEDURE("call "),
    STORED_PROCEDURE_WITH_RETURN("?= call "),
    ESCAPE("escape ");

    private final String _prefixStr;

    private Type(String prefixStr) {
      _prefixStr = prefixStr;
    }

    @Override
    public String toString() { return _prefixStr; }
  }

  private Type _type;
  private SqlObject _val;

  public JdbcEscape(Type type, SqlObject val) {
    _type = type;
    _val = val;
  }

  @Override
  public boolean hasParens() { return false; }

  @Override
  protected void collectSchemaObjects(ValidationContext vContext) {
    _val.collectSchemaObjects(vContext);
  }

  @Override
  public void appendTo(AppendableExt app) throws IOException {
    app.append("{").append(_type).append(_val).append("}");
  }

  /** @return a JDBC escaped value with the date portion of the given Date
   * @param d d*/
  public static JdbcEscape date(Date d) {
    java.sql.Date sqlDate = ((d instanceof java.sql.Date) ?
                             (java.sql.Date)d :
                             new java.sql.Date(d.getTime()));
    return new JdbcEscape(Type.DATE, new ValueObject(sqlDate));
  }

  /** @return a JDBC escaped value with the time portion of the given Date
   * @param d d */
  public static JdbcEscape time(Date d) {
    java.sql.Time sqlTime = ((d instanceof java.sql.Time) ?
                             (java.sql.Time)d :
                             new java.sql.Time(d.getTime()));
    return new JdbcEscape(Type.TIME, new ValueObject(sqlTime));
  }

  /** @return a JDBC escaped value with the date-time portion of the given
              Date (including milliseconds)
   @param d dd
   */
  public static JdbcEscape timestamp(Date d) {
    java.sql.Timestamp sqlTimestamp = ((d instanceof java.sql.Timestamp) ?
                                       (java.sql.Timestamp)d :
                                       new java.sql.Timestamp(d.getTime()));
    return new JdbcEscape(Type.TIMESTAMP, new ValueObject(sqlTimestamp));
  }

}
