package org.jsmth.data.sql;

import org.hibernate.MappingException;
import org.hibernate.internal.util.StringHelper;

import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * Created by mason on 15/12/16.
 */
public class TypeNames {
    /**
     * Holds default type mappings for a typeCode.  This is the non-sized mapping
     */
    private Map<Integer, String> defaults = new HashMap<Integer, String>();

    /**
     * Holds the weighted mappings for a typeCode.  The nested map is a TreeMap to sort its contents
     * based on the key (the weighting) to ensure proper iteration ordering during {@link #get(int, long, int, int)}
     */
    private Map<Integer, Map<Long, String>> weighted = new HashMap<Integer, Map<Long, String>>();

    /**
     * get default type name for specified type
     *
     * @param typeCode the type key
     *
     * @return 返回信息 the default type name associated with specified key
     *
     * @throws MappingException Indicates that no registrations were made for that typeCode
     */
    public String get(int typeCode) throws MappingException {
        final String result = defaults.get( typeCode );
        if ( result == null ) {
            throw new MappingException( "No Dialect mapping for JDBC type: " + typeCode );
        }
        return result;
    }

    /**
     * get type name for specified type and size
     *
     * @param typeCode the type key
     * @param size the SQL length
     * @param scale the SQL scale
     * @param precision the SQL precision
     * @return 返回信息 the associated name with smallest capacity size, if available and the default type name otherwise
     * @throws MappingException Indicates that no registrations were made for that typeCode
     */
    public String get(int typeCode, long size, int precision, int scale) throws MappingException {
        final Map<Long, String> map = weighted.get( typeCode );
        if ( map != null && map.size() > 0 ) {
            // iterate entries ordered by capacity to find first fit
            for ( Map.Entry<Long, String> entry: map.entrySet() ) {
                if ( size <= entry.getKey() ) {
                    return replace( entry.getValue(), size, precision, scale );
                }
            }
        }

        // if we get here one of 2 things happened:
        //		1) There was no weighted registration for that typeCode
        //		2) There was no weighting whose max capacity was big enough to contain size
        return replace( get( typeCode ), size, precision, scale );
    }

    private static String replace(String type, long size, int precision, int scale) {
        type = StringHelper.replaceOnce(type, "$s", Integer.toString(scale));
        type = StringHelper.replaceOnce( type, "$l", Long.toString( size ) );
        return StringHelper.replaceOnce( type, "$p", Integer.toString( precision ) );
    }

    /**
     * Register a weighted typeCode mapping
     *
     * @param typeCode the JDBC type code
     * @param capacity The capacity for this weighting
     * @param value The mapping (type name)
     */
    public void put(int typeCode, long capacity, String value) {
        Map<Long, String> map = weighted.get( typeCode );
        if ( map == null ) {
            // add new ordered map
            map = new TreeMap<Long, String>();
            weighted.put( typeCode, map );
        }
        map.put( capacity, value );
    }

    /**
     * Register a default (non-weighted) typeCode mapping
     *
     * @param typeCode the type key
     * @param value The mapping (type name)
     */
    public void put(int typeCode, String value) {
        defaults.put( typeCode, value );
    }
}
