package org.jsmth.data.service;

import org.apache.commons.lang.Validate;
import org.jsmth.data.dao.IEntityDao;
import org.jsmth.data.jdbc.Query;
import org.jsmth.data.sql.wrap.WhereWrap;
import org.jsmth.domain.SearchCondition;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.jorm.jdbc.EntityEventCallback;
import org.jsmth.jorm.jdbc.Event;
import org.jsmth.domain.Identifier;
import org.jsmth.page.CommonPage;
import org.jsmth.page.Page;
import org.jsmth.util.BeanUtils;
import org.jsmth.util.PropertyUtils;
import org.jsmth.util.ValueUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.persistence.MappedSuperclass;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Created by mason on 15/12/26.
 */
@MappedSuperclass
public class EntityService<KEY extends Serializable,
        MODEL extends Identifier<KEY>,
        ENTITYDAO extends IEntityDao<KEY, MODEL>> implements EntityEventCallback, IEntityService<KEY, MODEL, ENTITYDAO> {
    protected ENTITYDAO entityDao;
    protected Logger log = LoggerFactory.getLogger(this.getClass());
    protected EntityEventCallback entityEventCallback;

    public EntityService() {
    }

    @PostConstruct
    public void init() {
        Validate.notNull(entityDao, "entity dao must be set!");
        entityDao.setEntityEventCallback(this);
    }


    @Override
    public String getNameById(KEY id) {
        return getStrPropertyById(id, "name");
    }

    @Override
    public Object getPropertyById(KEY id, String propertyName) {
        return getPropertyById(id, propertyName, null);
    }

    @Override
    public Object getPropertyById(KEY id, String propertyName, Object defaultValue) {
        MODEL model = getById(id);
        if (model == null) {
            return defaultValue;
        }
        try {
            return PropertyUtils.getPropertyValue(model, propertyName);
        } catch (Exception ex) {
            ex.printStackTrace();
            return defaultValue;
        }
    }

    @Override
    public String getStrPropertyById(KEY id, String propertyName) {
        return getStrPropertyById(id, propertyName, "");
    }

    @Override
    public String getStrPropertyById(KEY id, String propertyName, String defaultValue) {
        Object value = getPropertyById(id, propertyName, defaultValue);
        return value.toString();
    }

    @Override
    public int getIntPropertyById(KEY id, String propertyName) {
        return getIntPropertyById(id, propertyName, 0);
    }

    @Override
    public int getIntPropertyById(KEY id, String propertyName, int defaultValue) {
        return ValueUtil.toInt(getPropertyById(id, propertyName, null), defaultValue);
    }

    @Override
    public long getLongPropertyById(KEY id, String propertyName) {
        return getLongPropertyById(id, propertyName, 0);
    }

    @Override
    public long getLongPropertyById(KEY id, String propertyName, long defaultValue) {
        return ValueUtil.toLong(getPropertyById(id, propertyName, null), defaultValue);
    }

    public Page<MODEL> pageByCondition(int pageNumber, int pageSize, boolean totalRecord, SearchCondition condition) {
        List<Object> params = new ArrayList<>();
        StringBuilder sql = new StringBuilder("1=1");
        buildWhere(params, sql, condition);
        return this.pageModels(pageNumber, pageSize, totalRecord, sql.toString(), params.toArray());
    }

    public List<MODEL> findByCondition(SearchCondition condition) {
        List<Object> params = new ArrayList<>();
        StringBuilder sql = new StringBuilder("1=1");
        buildWhere(params, sql, condition);
        return this.findModels(sql.toString(), params.toArray());
    }

    public MODEL getByCondition(SearchCondition condition) {
        List<MODEL> models = findByCondition(condition);
        if(models!=null && models.size()>0){
            return models.get(0);
        }
        return null;
    }

    public List<KEY> findIdsByCondition(SearchCondition condition) {
        List<Object> params = new ArrayList<>();
        StringBuilder sql = new StringBuilder("1=1");
        buildWhere(params, sql, condition);
        return findIds(sql.toString(), params.toArray());
    }

    public <K extends Serializable> List<K> findColumnsByCondition(Class<K> kClass, String columnName, SearchCondition condition) {
        List<Object> params = new ArrayList<>();
        StringBuilder sql = new StringBuilder("1=1");
        buildWhere(params, sql, condition);
        return findColumns(kClass, columnName, sql.toString(), params.toArray());
    }
    public <K extends Serializable> Page<K> pageColumnsByCondition(int pageNumber, int pageSize, boolean totalRecord,Class<K> kClass, String columnName, SearchCondition condition) {
        List<Object> params = new ArrayList<>();
        StringBuilder sql = new StringBuilder("1=1");
        buildWhere(params, sql, condition);
        return pageColumns(pageNumber,pageSize,totalRecord, kClass, columnName, sql.toString(), params.toArray());
    }

    protected void buildWhere(List params, StringBuilder sql, SearchCondition condition) {
        condition.buildWhere(params, sql);
    }


    protected Page<MODEL> pageModels(int pageNumber, int pageSize, boolean totalRecord, String where, Object... params) {
        return this.getEntityDao().pageModels(pageNumber, pageSize, totalRecord, where, params);
    }
    protected Page<MODEL> pageModels(int pageNumber, int pageSize, boolean totalRecord, Query query) {
        return this.getEntityDao().pageModels(pageNumber, pageSize, totalRecord,query);
    }

    protected List<MODEL> findModels( Query query){
        return this.getEntityDao().findModels(query);
    }

    protected List<MODEL> findModels(String where, Object... params) {
        return this.getEntityDao().findModels(where, params);
    }

    protected List<KEY> findIds(String where, Object... params) {
        return this.getEntityDao().findIds(getKeyClass(), where, params);
    }
    protected List<KEY> findIds(Query query) {
        return this.getEntityDao().findIds(getKeyClass(), query);
    }

    protected <K extends Serializable> List<K> findColumns(Class<K> kClass, String columnName, String where, Object... params) {
        return this.getEntityDao().findColumns(kClass, columnName, where, params);
    }
    protected <K extends Serializable> List<K> findColumns(Class<K> kClass, String columnName, Query query) {
        return this.getEntityDao().findColumns(kClass, columnName, query);
    }
    protected <K extends Serializable> Page<K> pageColumns(int pageNumber, int pageSize, boolean totalRecord,Class<K> kClass, String columnName, String where, Object... params) {
        return this.getEntityDao().pageFindColumn(kClass, columnName, where,pageNumber,pageSize, params);
    }
    protected <K extends Serializable> Page<K> pageColumns(int pageNumber, int pageSize, boolean totalRecord,Class<K> kClass, String columnName, Query query) {
        return this.getEntityDao().pageFindColumn(kClass, columnName, pageNumber,pageSize, query);
    }


    @Override
    public MODEL save(MODEL model) {
        return entityDao.save(model);
    }

    @Override
    public MODEL insert(MODEL model) {
        return entityDao.insert(model);
    }

    @Override
    public int update(MODEL model) {
        return entityDao.update(model);
    }

    protected int updateByIdAndWhere(MODEL model, String where, Object... params) {
        return entityDao.updateByIdAndWhere(model, where, params);
    }

    protected int updateByWhere(MODEL model, String where, Object... params) {
        return entityDao.updateByWhere(model, where, params);
    }


    @Override
    public int delete(MODEL model) {
        return entityDao.delete(model);
    }

    @Override
    public int deleteById(KEY id) {
        return entityDao.deleteById(id);
    }

    public MODEL getById(KEY id) {
        return entityDao.getById(id);
    }

    public MODEL getByIdOrNew(KEY id) throws SmthDataAccessException {
        return getByIdOrNew(id, true);
    }

    public MODEL getByIdOrNew(KEY id, boolean insert) throws SmthDataAccessException {
        return entityDao.getByIdOrNew(id, insert);
    }

    public List<KEY> findIds() {
        try {
            MODEL model = entityDao.getEntityClass().newInstance();
            return entityDao.findIds(model.getKeyClass(), "1=1");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new ArrayList<>();
    }

    public Page<KEY> pageIds(int pageNumber, int pageSize) {
        try {
            return entityDao.pageFindColumn(entityDao.getKeyClass(), "id", "1=1", pageNumber, pageSize);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return new CommonPage<>();
    }


    public List<MODEL> findAll() {
        return entityDao.findAll();
    }

    public Page<MODEL> findAll(int pageNumber, int pageSize) {
        return entityDao.findAll(pageNumber, pageSize);
    }

    public List<MODEL> findByIds(List<KEY> ids) throws SmthDataAccessException {
        if (ids == null || ids.size() == 0) {
            return new ArrayList<>();
        }
        return entityDao.findByIds(ids);
    }

    public List<MODEL> findByIdsOrNew(List<? extends KEY> ids) {
        if (ids == null || ids.size() == 0) {
            return new ArrayList<>();
        }
        return entityDao.findByIdsOrNew(ids);
    }

//
//    public void insertAll(Collection<MODEL> entities) throws SmthDataAccessException {
//        entityDao.insertAll(entities);
//    }


//    public int updateFields(MODEL entity, String... fieldNames) throws SmthDataAccessException {
//        return entityDao.up(entity, fieldNames);
//    }
//
//    protected int updateFeilds(boolean exclude, String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
//        return entityDao.updateFeilds(exclude, fieldNames, values, where, params);
//    }
//
//    protected int updateFeilds(String[] fieldNames, Object[] values, String where, Object... params) throws SmthDataAccessException {
//        return entityDao.updateFeilds(fieldNames, values, where, params);
//    }
//
//    protected int updateFeild(String fieldName, Object value, String where, Object... params) throws SmthDataAccessException {
//        return entityDao.updateFeild(fieldName, value, where, params);
//    }
//
//
//    protected int updateAllFeilds(boolean exclude, String[] fieldNames, Object[] values) throws SmthDataAccessException {
//        return entityDao.updateAllFeilds(exclude, fieldNames, values);
//    }
//
//    protected int updateAllFeilds(String[] fieldNames, Object[] values) throws SmthDataAccessException {
//        return entityDao.updateAllFeilds(fieldNames, values);
//    }
//
//    protected int updateAllFeild(String fieldName, Object value) throws SmthDataAccessException {
//        return entityDao.updateAllFeild(fieldName, value);
//    }
//
//
//    public int updateAll(Collection<MODEL> entities) throws SmthDataAccessException {
//        return entityDao.updateAll(entities);
//    }
//
//    public int updateFeildsAll(Collection<MODEL> entities, String... fieldNames) throws SmthDataAccessException {
//        return entityDao.updateFeildsAll(entities, fieldNames);
//    }

    protected Page<KEY> pageIds(int pageNumber, int pageSize, boolean totalRecord, String where, Object... params) {
        return entityDao.pageIds(getKeyClass(), pageNumber, pageSize, totalRecord, where, params);
    }

    public int deleteAll(Collection<MODEL> entities) throws SmthDataAccessException {
        return entityDao.deleteAll(entities);
    }

    public int deleteByIds(Collection<? extends KEY> ids) throws SmthDataAccessException {
        return entityDao.deleteByIds(ids);
    }

    protected int deleteModels(String where, Object... params) throws SmthDataAccessException {
        return entityDao.deleteModels(where, params);
    }

    public MODEL getModel(String where, Object... params) throws SmthDataAccessException {
        return this.entityDao.getModel(where, params);
    }
    protected MODEL getModel(Query query) throws SmthDataAccessException {
        return this.entityDao.getModel(query);
    }


    public ENTITYDAO getEntityDao() {
        return entityDao;
    }

    public void setEntityDao(ENTITYDAO entityDao) {
        this.entityDao = entityDao;
    }

    @Override
    public void batchInvokeEvent(Collection targets, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.batchInvokeEvent(targets, event);
        }
    }

    @Override
    public void invokeEvent(Object target, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.invokeEvent(target, event);
        }
    }

    @Override
    public Class<KEY> getKeyClass() {
        MODEL instance = BeanUtils.createInstance(entityDao.getEntityClass());
        return instance.getKeyClass();
    }

    @Override
    public <K extends Serializable> void setBeginKey(K beginKey, boolean isdelete, Class[] entityClasss) {
        this.entityDao.setBeginKey(beginKey, isdelete, entityClasss);
    }


}
