package org.jsmth.data.service;

import org.jsmth.cache.Cache;
import org.jsmth.cache.ICacheService;
import org.jsmth.data.dao.IEntityDao;
import org.jsmth.data.dao.IReadWriteEntityDao;
import org.jsmth.domain.Identifier;
import org.springframework.util.Assert;

import javax.annotation.PostConstruct;
import javax.persistence.MappedSuperclass;
import java.io.Serializable;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

/**
 * Created by mason on 15/12/26.
 */
@MappedSuperclass
public class EntityCacheableService<KEY extends Serializable, MODEL extends Identifier<KEY>,
        ENTITYDAO extends IEntityDao<KEY, MODEL>>
        extends EntityService<KEY, MODEL, ENTITYDAO>
        implements IEntityCacheableService<KEY, MODEL, ENTITYDAO> {

    protected boolean useReadWriteEntityDao = false;
    protected boolean useCache = true;
    protected ICacheService cacheService;
    protected int entityCacheMaxSize = 1000;
    protected int entityCacheMaxLiveSeconds = 3600;

    protected Cache entityCache;

    public EntityCacheableService() {
    }

    @PostConstruct
    @Override
    public void init() {
        super.init();
        Assert.notNull(cacheService, "cacheService must be set!");
        getEntityCache();
        useReadWriteEntityDao = (entityDao instanceof IReadWriteEntityDao);
    }

    protected Cache getEntityCache() {
        if (entityCache == null) {
            entityCache = cacheService.addCache(this.getClass().getName(), entityCacheMaxSize, entityCacheMaxLiveSeconds);
        }
        return entityCache;
    }

    protected List<MODEL> findModels(String where,Object...params){
        List<KEY> ids = findIds(where, params);
        return findByIds(ids);
    }
    protected List<KEY> findIds(String where,Object...params){
        return this.getEntityDao().findIds(this.getEntityDao().getKeyClass(),where,params);
    }


    @Override
    public MODEL insert(MODEL model) {
        model = super.insert(model);
        doDeleteEntityCache(model);
        if (model.isIdModified()) {
            getById(model.getIdentifier());
        }
        return model;
    }

    @Override
    public int update(MODEL model) {
        int update = super.update(model);
        doDeleteEntityCache(model);
        if (model.isIdModified()) {
            getById(model.getIdentifier());
        }
        return update;
    }

    @Override
    public int delete(MODEL model) {
        int update = super.delete(model);
        doDeleteEntityCache(model);
        return update;
    }

    @Override
    public int deleteById(KEY id) {
        int update = super.deleteById(id);
        doDeleteEntityCacheById(id);
        return update;
    }

    @Override
    public MODEL getById(KEY id) {
        return getById(id, true);
    }

    public MODEL getById(KEY id, boolean useMasterExplicit) {
        if (useMasterExplicit)
            return EntityCacheHelper.getById(useCache, entityCacheMaxLiveSeconds, entityDao.getEntityClass(), entityDao.getJdbcDao(), getEntityCache(), id);
        else
            return EntityCacheHelper.getById(useCache, entityCacheMaxLiveSeconds, entityDao.getEntityClass(), ((IReadWriteEntityDao) entityDao).getSlaveJdbcDao(), getEntityCache(), id);
    }



    public ICacheService getCacheService() {
        return cacheService;
    }

    public void setCacheService(ICacheService cacheService) {
        this.cacheService = cacheService;
    }

    protected void doDeleteAllEntityCache() {
        getEntityCache().clear();
    }

    protected void doDeleteEntityCache(MODEL entity) {
        EntityCacheHelper.onEntityDelete(getEntityCache(), entity);
    }

    protected void doDeleteEntityCacheById(KEY id) {
        EntityCacheHelper.onEntityIdsDelete(getEntityCache(), id);
    }

    protected void doDeleteEntityCache(Collection<MODEL> entities) {
        EntityCacheHelper.onEntityDelete(getEntityCache(), entities.toArray(new Identifier[entities.size()]));
    }

    protected void doDeleteEntityCacheByIds(Collection<KEY> ids) {
        List<KEY> keys = new LinkedList<KEY>(ids);
        List<MODEL> models = this.getEntityDao().findByIds(keys);
        doDeleteEntityCache(models);
    }

    protected void doDeleteEntityCacheByWhere(String where, Object... params) {
        List<MODEL> models = this.getEntityDao().findModels(where, params);
        doDeleteEntityCache(models);
    }

    protected void doUpdateEntityCache(MODEL entity) {
        if (useCache) {
            EntityCacheHelper.onEntityUpdate(entityCacheMaxLiveSeconds, getEntityCache(), entity);
        }
    }

    protected void doUpdateEntityCache(Collection<MODEL> entities) {
        if (useCache) {
            EntityCacheHelper.onEntityUpdate(entityCacheMaxLiveSeconds, getEntityCache(), entities.toArray(new Identifier[entities.size()]));
        }
    }

    protected void doUpdateEntityCacheByIds(Collection<KEY> ids) {
        List<KEY> keys = new LinkedList<KEY>(ids);
        List<MODEL> models = this.getEntityDao().findByIds(keys);
        doUpdateEntityCache(models);
    }

    protected void doUpdateEntityCacheByWhere(String where, Object... params) {
        List<MODEL> models = this.getEntityDao().findModels(where, params);
        doUpdateEntityCache(models);
    }
}
