package org.jsmth.data.schema;

import org.apache.commons.lang.Validate;
import org.jsmth.jorm.jdbc.Column;
import org.jsmth.jorm.jdbc.Event;
import org.jsmth.jorm.jdbc.Index;
import org.jsmth.jorm.jdbc.JPAHelper;
import org.jsmth.util.PropertyUtils;
import org.jsmth.util.ReflectUtil;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;

/**
 * Created by mason on 15/12/29.
 */
public class ObjectTableMeta<T> {

    private static final Map<Class, ObjectTableMeta> tableCache = new HashMap<Class, ObjectTableMeta>();

    @SuppressWarnings({"unchecked"})
    public static <T> ObjectTableMeta<T> getTable(Class<T> clazz) {
        ObjectTableMeta<T> ret = tableCache.get(clazz);
        if (ret == null) {
            synchronized (tableCache) {
                ret = tableCache.get(clazz);
                if (ret == null) {
                    ret = new ObjectTableMeta<T>(clazz);
                }
                tableCache.put(clazz, ret);
            }
        }
        return ret;
    }

    Class<T> clazz;
    String tableName;
    Column idColumn;
    Class keyClass;


    Set<Column> columns = new LinkedHashSet<Column>();
    Map<String, Column> columnsByColumnName = new LinkedHashMap<String, Column>();
    Map<String, Column> columnsByFieldName = new LinkedHashMap<String, Column>();
    Map<Event, Method> events = new HashMap<Event, Method>(3);
    Map<String, Index> indexes = new LinkedHashMap<String, Index>();

//    Map<String, PropertyDescriptor> descriptors = new HashMap<String, PropertyDescriptor>();


    public ObjectTableMeta(Class<T> clazz) {
        Validate.isTrue(JPAHelper.isEntity(clazz), "entity must annotated by @Entity");
        this.clazz = clazz;
        this.tableName = JPAHelper.getTableName(clazz);
        this.keyClass = (Class) ReflectUtil.getGenericClassParameterizedType(clazz);

        List<Field> fields = JPAHelper.getJPAFields(clazz);

        for (Field field : fields) {
            List<Column> columnList = getColumnsByField(field);
            for (Column column : columnList) {
                if (column.isIdentifier()) {
                    if (idColumn == null)
                        idColumn = column;
                    else
                        throw new IllegalArgumentException("Two or more ID field.");
                }
                columns.add(column);
            }
        }

        //子类属性重载部分
        Map<String, javax.persistence.Column> override = JPAHelper.getColumnOverride(clazz);

        for (Column column : columns) {
            String fieldName = column.getFieldName();
            javax.persistence.Column ann = override.get(fieldName);
            if (ann != null) {
                column.setColumnName(ann.name());
            }
            //数据表名可以全都小写
            columnsByColumnName.put(column.getColumnName().toLowerCase(), column);
            //属性名不能小写
            columnsByFieldName.put(column.getFieldName(), column);
        }
        //事件部分
        this.events.putAll(JPAHelper.getEventMap(clazz));

        //索引部分
        Set<Index> indexes = JPAHelper.getIndexFromClass(clazz);
        for (Index index : indexes) {
            this.indexes.put(index.getName(), index);
        }
        for (Field field : fields) {
            Index index = JPAHelper.getIndexFromField(field);
            if (index != null) {
                this.indexes.put(index.getName(), index);
            }
        }

        Validate.notNull(idColumn, "id column must not be null");
    }

    List<Column> getColumnsByField(Field field) {
        return getColumnsByField(field, false, null);
    }

    List<Column> getColumnsByField(Field field, boolean isEmbeded, FieldEmbeddedSchema fieldEmbeddedSchema) {
        List<Column> list = new ArrayList<>();
        if (!JPAHelper.isEmbedded(field)) {
            Column col = new Column(field, keyClass);
            if (isEmbeded) {
                col.setFieldEmbeddedSchema(fieldEmbeddedSchema);
            }
            col.setEmbedded(isEmbeded);
            list.add(col);
        } else {
            FieldEmbeddedSchema sub = new FieldEmbeddedSchema();
            sub.setField(field);
            sub.setParent(fieldEmbeddedSchema);
            List<Field> emfields = JPAHelper.getJPAFields(field.getType());
            for (Field emfield : emfields) {
                List<Column> list1 = getColumnsByField(emfield, true, sub);
                list.addAll(list1);
            }
        }
        return list;
    }

    public Object getFieldValue(String fieldName, Object instance) {
        Column column = columnsByFieldName.get(fieldName);
        Validate.notNull(column, "no mapped field for " + fieldName);
        return getFieldValue(column,instance);
    }

    public Object getFieldValue(Column column, Object instance) {
        Validate.notNull(column, "column is null" );

        if (column.isEmbedded()) {
            String fieldName1 = column.getFieldEmbeddedSchema().getFieldName();
            instance = PropertyUtils.getPropertyValue(instance, fieldName1);
            Validate.notNull(instance, "embedded field value is null" + fieldName1);
        }
        Object propertyValue = PropertyUtils.getPropertyValue(instance, column.getFieldName());
        if (propertyValue == null)
            return null;
        if (column.isEnumerate()) {
            Enum value = (Enum) propertyValue;
            if (value == null) {
                throw new IllegalArgumentException("Enum Field [" + column.getField() + "] can not be null .");
            }
            if (column.isUseOrdinal()) {
                propertyValue = value.ordinal();
            } else {
                propertyValue = value.name();
            }
        }
        return propertyValue;
    }

    public void invokeEvent(Object target, Event event) {
        if (target == null) return;

        Method method = events.get(event);
        if (method != null) {
            try {
                method.invoke(target);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void batchInvokeEvent(Collection targets, Event event) {
        if (targets.isEmpty())
            return;

        if (!this.events.containsKey(event))
            return;

        Method method = events.get(event);

        for (Object target : targets) {
            if (target == null) continue;

            try {
                method.invoke(target);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } catch (InvocationTargetException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public List<Column> selectColumns(boolean excludeId, boolean exclude, String... fieldNames) {
        LinkedList<Column> items = new LinkedList<>();
        Set<String> fields = new LinkedHashSet<>();
        if (fieldNames != null) {
            for (String fieldName : fieldNames) {
                fields.add(fieldName);
            }
        }
        if (exclude) {
            for (Column column : columns) {
                if (column.isIdentifier() && excludeId) {
                    continue;
                }
                if (fields.contains(column.getFieldName())) {
                    continue;
                }
                items.add(column);
            }
        } else {
            Validate.notEmpty(fieldNames, "select clause should have one field");
            for (Column column : columns) {
                if (column.isIdentifier() && excludeId) {
                    continue;
                }
                if (fields.contains(column.getFieldName())) {
                    items.add(column);
                }
            }
        }
        return items;
    }

    public Class<T> getClazz() {
        return clazz;
    }

    public void setClazz(Class<T> clazz) {
        this.clazz = clazz;
    }

    public String getTableName() {
        return tableName;
    }

    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    public Column getIdColumn() {
        return idColumn;
    }

    public void setIdColumn(Column idColumn) {
        this.idColumn = idColumn;
    }

    public Class getKeyClass() {
        return keyClass;
    }

    public void setKeyClass(Class keyClass) {
        this.keyClass = keyClass;
    }

    public Set<Column> getColumns() {
        return columns;
    }

    public void setColumns(Set<Column> columns) {
        this.columns = columns;
    }

    public Map<String, Column> getColumnsByColumnName() {
        return columnsByColumnName;
    }

    public void setColumnsByColumnName(Map<String, Column> columnsByColumnName) {
        this.columnsByColumnName = columnsByColumnName;
    }

    public Map<String, Column> getColumnsByFieldName() {
        return columnsByFieldName;
    }

    public void setColumnsByFieldName(Map<String, Column> columnsByFieldName) {
        this.columnsByFieldName = columnsByFieldName;
    }

    public Map<Event, Method> getEvents() {
        return events;
    }

    public void setEvents(Map<Event, Method> events) {
        this.events = events;
    }

    public Map<String, Index> getIndexes() {
        return indexes;
    }

    public void setIndexes(Map<String, Index> indexes) {
        this.indexes = indexes;
    }

    public Column getColumnByColumnName(String name) {
        return columnsByColumnName.get(name);
    }

    public Column getColumnByFieldName(String name) {
        return columnsByFieldName.get(name);
    }
}
