package org.jsmth.data.redis;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by mason on 2017/12/7.
 */
public class RedisQueueDao {
    private String host = "127.0.0.1";
    private int port = 6379;
    private String password = "";

    boolean testOnBorrow = true;
    boolean testOnReturn = true;
    boolean testWhileIdle = true;
    int maxActive = 5000;
    int maxIdle = 256;
    long maxWait = 5000L;
    long minEvictableIdleTimeMillis = 60000L;
    long timeBetweenEvictionRunsMillis = 3000L;
    int numTestsPerEvictionRun = -1;
    int timeout = 60000;

    private JedisPool jedisPool;

    void init() {
        if (jedisPool != null) {
            return;
        }
        //Configuration自行写的配置文件解析类,继承自Properties
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxActive(maxActive);
        config.setMaxIdle(maxIdle);
        config.setMaxWait(maxWait);
        config.setTestOnBorrow(testOnBorrow);
        config.setTestOnReturn(testOnReturn);
        config.setTestWhileIdle(testWhileIdle);
        config.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);
        config.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        config.setNumTestsPerEvictionRun(numTestsPerEvictionRun);
        jedisPool = new JedisPool(config, host, port, timeout);
    }


    private void close(Jedis jedis) {
        try {
            jedisPool.returnResource(jedis);
        } catch (Exception e) {
            if (jedis.isConnected()) {
                jedis.quit();
                jedis.disconnect();
                jedisPool = null;
            }
        }
    }

    //<editor-fold desc="String key">

    /**
     * 获取数据
     *
     * @param key ff
     * @return ff
     */
    public String get(String key) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.get(key);
            }
        }, "", key);
    }

    public String set(String key, String value) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.set(key, value);
            }
        }, "", key);
    }

    public long set(String key, String value, int time) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                jedis.set(key, value);
                return jedis.expire(key, time);
            }
        }, 0L, key);
    }

    public long hset(byte[] key, byte[] field, byte[] value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.hset(key, field, value);
            }
        }, 0L, key);
    }

    public byte[] get(byte[] key) {
        return exec(new CommandTemplate<byte[]>() {
            @Override
            public byte[] exec(Jedis jedis, Object... params) {
                return jedis.get(key);
            }
        }, null, key);
    }

    public String set(byte[] key, byte[] value) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.set(key, value);
            }
        }, "", key);
    }

    public long set(byte[] key, byte[] value, int time) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                jedis.set(key, value);
                return jedis.expire(key, time);
            }
        }, 0L, key);
    }


    public long hset(String key, String field, String value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.hset(key, field, value);
            }
        }, 0L, key);
    }

    /**
     * 获取数据
     *
     * @param field dd
     * @param key ff
     * @return ff
     */
    public String hget(String key, String field) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.hget(key, field);
            }
        }, "", key);
    }


    public long hdel(String key, String field) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.hdel(key, field);
            }
        }, 0L, key);
    }

    /**
     * 存储REDIS队列 顺序存储
     *
     * @param key   reids键名
     * @param value 键值
     * @return dd
     */
    public long lpush(String key, String value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.lpush(key, value);
            }
        }, 0L, key);
    }

    /**
     * 存储REDIS队列 反向存储
     *
     * @param key   reids键名
     * @param value 键值
     * @return  ff
     */
    public long rpush(String key, String value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.rpush(key, value);
            }
        }, 0L, key);
    }

    /**
     * 将列表 source 中的最后一个元素(尾元素)弹出，并返回给客户端
     *
     * @param key         reids键名
     * @param destination 键值
     * @return dd
     */
    public String rpoplpush(String key, String destination) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.rpoplpush(key, destination);
            }
        }, "", key);
    }

    /**
     * 获取队列数据
     *
     * @param key 键名
     * @return ff
     */
    public List lpopList(String key) {
        return exec(new CommandTemplate<List>() {
            @Override
            public List exec(Jedis jedis, Object... params) {
                return jedis.lrange(key, 0, -1);
            }
        }, null, key);
    }

    /**
     * 获取队列数据
     *
     * @param key 键名
     * @return ff
     */
    public String rpop(String key) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.rpop(key);
            }
        }, "", key);
    }

    public String hmset(Object key, Map hash) {
        return exec(new CommandTemplate<String>() {
            @Override
            public String exec(Jedis jedis, Object... params) {
                return jedis.hmset(key.toString(), hash);
            }
        }, "", key);
    }

    public long hmset(Object key, Map hash, int time) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                jedis.hmset(key.toString(), hash);
                return jedis.expire(key.toString(), time);
            }
        }, 0L, key);
    }

    public List hmget(Object key, String... fields) {
        return exec(new CommandTemplate<List>() {
            @Override
            public List exec(Jedis jedis, Object... params) {
                return jedis.hmget(key.toString(), fields);
            }
        }, null, key);
    }

    public Set hkeys(String key) {
        return exec(new CommandTemplate<Set>() {
            @Override
            public Set exec(Jedis jedis, Object... params) {
                return jedis.hkeys(key);
            }
        }, null, key);
    }

    public List lrange(String key, int from, int to) {
        return exec(new CommandTemplate<List>() {
            @Override
            public List exec(Jedis jedis, Object... params) {
                return jedis.lrange(key, from, to);
            }
        }, null, key);
    }

    public Map hgetAll(String key) {
        return exec(new CommandTemplate<Map>() {
            @Override
            public Map exec(Jedis jedis, Object... params) {
                return jedis.hgetAll(key);
            }
        }, null, key);
    }

    public long del(String key) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.del(key);
            }
        }, 0L, key);
    }

    public long llen(String key) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.llen(key);
            }
        }, 0L, key);
    }
    //</editor-fold>

    /*88888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888*/


    //<editor-fold desc="byte key">

    /**
     * 获取数据
     *
     * @param key ff
     * @param field dd
     * @return ff
     */
    public byte[] hget(byte[] key, byte[] field) {
        return exec(new CommandTemplate<byte[]>() {
            @Override
            public byte[] exec(Jedis jedis, Object... params) {
                return jedis.hget(key, field);
            }
        }, null, key);
    }

    public long hdel(byte[] key, byte[] field) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.hdel(key, field);
            }
        }, 0L, key);
    }

    /**
     * 存储REDIS队列 顺序存储
     *
     * @param key   reids键名
     * @param value 键值
     * @return dd
     */
    public long lpush(byte[] key, byte[] value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.lpush(key, value);
            }
        }, 0L, key);
    }

    /**
     * 存储REDIS队列 反向存储
     *
     * @param key   reids键名
     * @param value 键值
     * @return dd
     */
    public long rpush(byte[] key, byte[] value) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.rpush(key, value);
            }
        }, 0L, key);
    }

    /**
     * 将列表 source 中的最后一个元素(尾元素)弹出，并返回给客户端
     *
     * @param key         reids键名
     * @param destination 键值
     * @return dd
     */
    public byte[] rpoplpush(byte[] key, byte[] destination) {
        return exec(new CommandTemplate<byte[]>() {
            @Override
            public byte[] exec(Jedis jedis, Object... params) {
                return jedis.rpoplpush(key, destination);
            }
        }, null, key);
    }

    /**
     * 获取队列数据
     *
     * @param key 键名
     * @return ff
     */
    public List lpopList(byte[] key) {
        return exec(new CommandTemplate<List>() {
            @Override
            public List exec(Jedis jedis, Object... params) {
                return jedis.lrange(key, 0, -1);
            }
        }, null, key);
    }

    /**
     * 获取队列数据
     *
     * @param key 键名
     * @return ff
     */
    public byte[] rpop(byte[] key) {
        return exec(new CommandTemplate<byte[]>() {
            @Override
            public byte[] exec(Jedis jedis, Object... params) {
                return jedis.rpop(key);
            }
        }, null, key);
    }


    public List lrange(byte[] key, int from, int to) {
        return exec(new CommandTemplate<List>() {
            @Override
            public List exec(Jedis jedis, Object... params) {
                return jedis.lrange(key, from, to);
            }
        }, null, key);
    }

    public Map hgetAll(byte[] key) {
        return exec(new CommandTemplate<Map>() {
            @Override
            public Map exec(Jedis jedis, Object... params) {
                return jedis.hgetAll(key);
            }
        }, null, key);
    }

    public long del(byte[] key) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.del(key);
            }
        }, 0L, key);
    }

    public long llen(byte[] key) {
        return exec(new CommandTemplate<Long>() {
            @Override
            public Long exec(Jedis jedis, Object... params) {
                return jedis.llen(key);
            }
        }, 0L, key);
    }
    //</editor-fold>


    public <R> R exec(CommandTemplate<R> template, R defaultValue, Object... params) {
        long len = 0;
        Jedis jedis = null;
        try {
            init();
            jedis = jedisPool.getResource();
            return template.exec(jedis, params);
        } catch (Exception e) {
            //释放redis对象
            jedisPool.returnBrokenResource(jedis);
            e.printStackTrace();
        } finally {
            //返还到连接池
            close(jedis);
        }
        return defaultValue;
    }

    interface CommandTemplate<R> {
        R exec(Jedis jedis, Object... params);
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public boolean isTestOnBorrow() {
        return testOnBorrow;
    }

    public void setTestOnBorrow(boolean testOnBorrow) {
        this.testOnBorrow = testOnBorrow;
    }

    public boolean isTestOnReturn() {
        return testOnReturn;
    }

    public void setTestOnReturn(boolean testOnReturn) {
        this.testOnReturn = testOnReturn;
    }

    public boolean isTestWhileIdle() {
        return testWhileIdle;
    }

    public void setTestWhileIdle(boolean testWhileIdle) {
        this.testWhileIdle = testWhileIdle;
    }

    public int getMaxActive() {
        return maxActive;
    }

    public void setMaxActive(int maxActive) {
        this.maxActive = maxActive;
    }

    public int getMaxIdle() {
        return maxIdle;
    }

    public void setMaxIdle(int maxIdle) {
        this.maxIdle = maxIdle;
    }

    public long getMaxWait() {
        return maxWait;
    }

    public void setMaxWait(long maxWait) {
        this.maxWait = maxWait;
    }

    public long getMinEvictableIdleTimeMillis() {
        return minEvictableIdleTimeMillis;
    }

    public void setMinEvictableIdleTimeMillis(long minEvictableIdleTimeMillis) {
        this.minEvictableIdleTimeMillis = minEvictableIdleTimeMillis;
    }

    public long getTimeBetweenEvictionRunsMillis() {
        return timeBetweenEvictionRunsMillis;
    }

    public void setTimeBetweenEvictionRunsMillis(long timeBetweenEvictionRunsMillis) {
        this.timeBetweenEvictionRunsMillis = timeBetweenEvictionRunsMillis;
    }

    public int getNumTestsPerEvictionRun() {
        return numTestsPerEvictionRun;
    }

    public void setNumTestsPerEvictionRun(int numTestsPerEvictionRun) {
        this.numTestsPerEvictionRun = numTestsPerEvictionRun;
    }

    public int getTimeout() {
        return timeout;
    }

    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }
}
