package org.jsmth.data.redis;

import org.jsmth.data.keydao.KeyEntityDao;
import org.jsmth.domain.Identifier;
import org.jsmth.util.ReflectUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.*;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.JedisShardInfo;

import java.io.Serializable;
import java.lang.reflect.Type;

/**
 * Created by mason on 15/12/26.
 */
public class RedisEntityDao<KEY extends Serializable, MODEL extends Identifier<KEY>>
        extends KeyEntityDao<KEY,MODEL> implements IRedisEntityDao<KEY,MODEL>{
    protected Logger log = LoggerFactory.getLogger(this.getClass());

    int maxIdle=300;
    int maxActive=600;
    int maxWait=1000 ;
    boolean testOnBorrow=true;
    String host="";
    int port=6379;
    String password="";


    RedisTemplate<KEY,MODEL> redisTemplate;


    public void init(){
        JedisShardInfo shardInfo=new JedisShardInfo(host,port);
        shardInfo.setPassword(password);
        JedisPoolConfig jedisPoolConfig=new JedisPoolConfig();
        jedisPoolConfig.setMaxIdle(maxIdle);
        jedisPoolConfig.setMaxActive(maxActive);
        jedisPoolConfig.setMaxWait(maxWait);
        jedisPoolConfig.setTestOnBorrow(testOnBorrow);
        JedisConnectionFactory factory=new JedisConnectionFactory(shardInfo);
        factory.setPoolConfig(jedisPoolConfig);
        redisTemplate=new RedisTemplate<>() ;
        redisTemplate.setConnectionFactory(factory);
        redisTemplate.setKeySerializer(RedisKeySerializerFactory.createRedisSerializer(getKeyClassType()));
        redisTemplate.setValueSerializer(new JasonEntityRedisSerializer<KEY,MODEL>());
    }

    public Type getKeyClassType(){
        return ReflectUtil.getGenericClassParameterizedType(this.getClass(),0);
    }
    public Type getEntityClassType(){
        return ReflectUtil.getGenericClassParameterizedType(this.getClass(), 1);
    }

    @Override
    public int put(final MODEL model) {
        ValueOperations<KEY, MODEL> operations = redisTemplate.opsForValue();
        operations.set(model.getIdentifier(), model);
        return 1;
    }

    String buildKey(MODEL model){
        return buildKey(model.getClass().getName(), model.getIdentifier());
    }
    String buildKey(String className,KEY id){
        StringBuilder builder=new StringBuilder();

        builder.append("k");
        builder.append(className);
        builder.append(id);
        return builder.toString();
    }

    @Override
    public MODEL get(KEY id) {
        ValueOperations<KEY, MODEL> operations = redisTemplate.opsForValue();
        return operations.get(id);
    }

    @Override
    public int remove(KEY id) {
        redisTemplate.delete(id);
        return 1;
    }

    protected RedisTemplate<KEY, MODEL> getRedisTemplate() {
        return redisTemplate;
    }

    protected void setRedisTemplate(RedisTemplate<KEY, MODEL> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    public int getMaxIdle() {
        return maxIdle;
    }

    public void setMaxIdle(int maxIdle) {
        this.maxIdle = maxIdle;
    }

    public int getMaxActive() {
        return maxActive;
    }

    public void setMaxActive(int maxActive) {
        this.maxActive = maxActive;
    }

    public int getMaxWait() {
        return maxWait;
    }

    public void setMaxWait(int maxWait) {
        this.maxWait = maxWait;
    }

    public boolean isTestOnBorrow() {
        return testOnBorrow;
    }

    public void setTestOnBorrow(boolean testOnBorrow) {
        this.testOnBorrow = testOnBorrow;
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }
}
