package org.jsmth.data.jdbc;

import org.hibernate.annotations.Where;
import org.jsmth.data.sql.SqlQueryType;
import org.jsmth.domain.Identifier;
import org.jsmth.data.dialect.Dialect;
import org.jsmth.data.schema.ObjectTableMeta;
import org.jsmth.data.sql.EntityQuery;
import org.jsmth.data.sql.wrap.*;
import org.jsmth.exception.SmthDataAccessException;
import org.jsmth.exception.SmthException;
import org.jsmth.jorm.jdbc.*;
import org.jsmth.page.Page;
import org.jsmth.util.ObjectUtils;
import org.springframework.jdbc.core.namedparam.EmptySqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import javax.persistence.MappedSuperclass;
import javax.sql.DataSource;
import java.io.Serializable;
import java.util.*;

/**
 * Created by mason on 15/12/26.
 */
@MappedSuperclass
public class JdbcDao extends TableJdbcDao implements IJdbcDao {
    //<editor-fold desc="JdbcDao">
    JdbcDao(DataSource dataSource, Dialect dialect) {
        super(dataSource, dialect);
    }

    public JdbcDao(DataSource dataSource) {
        super(dataSource);
    }

    public JdbcDao(SmthDataSource dataSource) {
        super(dataSource);
    }

    public JdbcDao(DataSource dataSource, String dialectString) {
        super(dataSource, dialectString);
    }
    //</editor-fold>

    //<editor-fold desc="save">
    /**
     * 保存实体。如果实体不存在则添加
     *
     * @param entity ff
     * @param <T> ff
     * @return 返回信息
     */
    public <T extends Identifier> T save(T entity) throws SmthDataAccessException {
        if (entity.isIdModified()) {
            this.updateModelById(entity);
        } else {
            this.insert(entity);
        }
        return entity;
    }
    //</editor-fold>
    //<editor-fold desc="insert">
    public <T extends Identifier> T insert(T model) {
        if (model == null) {
            throw new SmthException("model is null");
        }
        ObjectTableMeta<? extends Identifier> table = ObjectTableMeta.getTable(model.getClass());
        table.invokeEvent(model, Event.PreInsert);
        InsertFieldWrap insertFieldWrap = new InsertFieldWrap();
        InsertValueWrap insertValueWrap = new InsertValueWrap();
        boolean isAuto=false;
        for (Column column : table.getColumns()) {
            if (column.isIdentifier() && column.isIdAutoIncrease()) {
                if (!model.isIdModified()) {
                    isAuto=true;
                    continue;
                }
            }
            BeanPropertySqlParameterSourceEx ex = new BeanPropertySqlParameterSourceEx(model);
            Object value = ex.getValue(column.getFieldName());
            if (value == null) {
                continue;
            }
            insertFieldWrap.f(column.getColumnName());
            insertValueWrap.v(column.getColumnName(), value);
        }
        if(isAuto){
            Number num=insertBackKey(model.getClass(), insertFieldWrap, insertValueWrap);
            if(num!=null){
                if (table.getIdColumn().getType() == ColumnType.Int)
                    model.setIdentifier(num.intValue());
                else if (table.getIdColumn().getType() == ColumnType.Long)
                    model.setIdentifier(num.longValue());
                else
                    throw new IllegalArgumentException("GeneratedKeyHolder is support int or long");
            }
        }else{
            int r=insert(model.getClass(), insertFieldWrap, insertValueWrap);
            if(r<1){
                return null;
            }
        }
        table.invokeEvent(model, Event.PostInsert);
        return model;

    }
    //</editor-fold>
    //<editor-fold desc="update">
    public  <T> int updateModelById(T entity,String... ommitFieldNames) {
        return updateModelById(entity, true, ommitFieldNames);
    }
    public  <T> int updateModelById(T entity, boolean exclude,String... fieldNames) {
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(entity.getClass());
        WhereWrap whereWrap=new WhereWrap();
        Object idVal = table.getFieldValue(table.getIdColumn().getFieldName(), entity);
        whereWrap.wEqual(table.getIdColumn().getFieldName(),idVal);
        return updateModel(entity, whereWrap, exclude, fieldNames);
    }

    public  <T> int updateModelByIdAndWhere(T entity,String where ,Object ... params) {
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(entity.getClass());
        WhereWrap whereWrap=new WhereWrap();
        Object idVal = table.getFieldValue(table.getIdColumn().getFieldName(), entity);

        if(where!=null && where.length()>0){
            where+=" and ";
        }
        where+=" id=? ";
        List<Object> ps=new ArrayList<>();

        if(params!=null){
            for (Object param : params) {
                ps.add(param);
            }
        }
        ps.add(idVal);
        return updateModelByWhere(entity, where,ps.toArray());
    }
    public  <T> int updateModelByWhere(T entity,String where ,Object ... params) {
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW(where,params);
        String[] fieldNames=new String[]{};
        return updateModel(entity, whereWrap,true,fieldNames);
    }

    public  <T> int updateModel(T entity, WhereWrap whereWrap,boolean exclude,String... fieldNames) {
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(entity.getClass());
        List<Column> columns = table.selectColumns(true, exclude, fieldNames);
        UpdateFieldWrap updateFieldWrap=new UpdateFieldWrap();
        invokeEvent(table, entity, Event.PreUpdate);
        for (Column column : columns) {
            Object val = table.getFieldValue(column, entity);
            if(val==null){
                continue;
            }
            updateFieldWrap.u(column.getFieldName(),val);
        }
        int update = updateModel(entity, updateFieldWrap, whereWrap);
        invokeEvent(table, entity, Event.PostUpdate);
        return update;
    }

    //</editor-fold>
    //<editor-fold desc="delete">
    public int deleteById(Class eClass, Serializable id){
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(eClass);
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW("id=?",id);
        return delete(eClass, whereWrap);
    }
    public int deleteAll(Class eClass){
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(eClass);
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW("1=?",1);
        return delete(eClass, whereWrap);
    }
    public<K extends Serializable, T extends Identifier<K>>  int deleteByIds(Class<T> eClass, Collection<K> ids){
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(eClass);
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.wIn(table.getIdColumn().getFieldName(), ids);
        return delete(eClass, whereWrap);
    }
    public <K extends Serializable> int deleteAll(Collection entitys){
        if(entitys==null || entitys.size()==0)
            return 0;
        Class tClass=null;
        for (Object entity : entitys) {
            tClass = entity.getClass();
            break;
        }
        ObjectTableMeta table = ObjectTableMeta.getTable(tClass);
        List<K> ids = ObjectUtils.getFields(entitys, table.getIdColumn().getFieldName());
        return deleteByIds(tClass,ids);
    }
    public <T extends Identifier> int delete(T entity){
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(entity.getClass());
        invokeEvent(table, entity, Event.PreDelete);
        int update = deleteById(entity.getClass(), entity.getIdentifier());
        invokeEvent(table, entity, Event.PostDelete);
        return update;
    }
    //</editor-fold>
    //<editor-fold desc="query">
    public <K,T> List<T> queryByIds(Class<T> entityClass, List<K> ids)  {
        ObjectTableMeta<T> table =ObjectTableMeta.getTable(entityClass);
        return queryByColumns(entityClass, table.getIdColumn().getFieldName(), ids.toArray());
    }
    public <K extends Serializable, T extends Identifier<K>> T getOrCreateById(Class<T> entityClass, K id){
        T t = getById(entityClass, id);
        if(t==null){
            try {
                t=entityClass.newInstance();
                t.setIdentifier(id);
                t=insert(t);
            } catch (Exception e) {
                logger.error("getOrCreateById",e);
                return null;
            }
        }
        return t;
    }

    public <T> List<T> query(String sql,Class<T> tClass)  {
        return jdbcTemplate.query(sql, (SqlParameterSource) EmptySqlParameterSource.INSTANCE, new JPARowMapper<T>(tClass));
    }
    public <K extends Serializable, T extends Identifier> T getById(Class<T> mClass, K id) {
        ObjectTableMeta<? extends Object> table = ObjectTableMeta.getTable(mClass);
        WhereWrap whereWrap = new WhereWrap();
        whereWrap.placeholderW(table.getIdColumn().getFieldName()+"=?", id);
        return queryForObject(mClass, whereWrap);
    }

    //</editor-fold>

    //<editor-fold desc="for queryByColumns">
    public <T> List<T> queryByColumns(Class<T> entityClass, String fieldName, Object... columnValues) {
        StringBuilder where=new StringBuilder();
        for (Object columnValue : columnValues) {
            if(where.length()>0){
                where.append(",");
            }
            where.append("?");
        }
        where.insert(0,fieldName +" in (");
        where.append(")");
        return query(entityClass,"*",where.toString(),columnValues);
    }
    public <T> Page<T> queryPageByColumns(Class<T> entityClass,int pageNumber,int pageSize, String fieldName, Object... columnValues) {
        StringBuilder where=new StringBuilder();
        for (Object columnValue : columnValues) {
            if(where.length()>0){
                where.append(",");
            }
            where.append("?");
        }
        where.insert(0,fieldName +" in (");
        where.append(")");
        return queryPage(entityClass,pageNumber,pageSize,true,"*",where.toString(),columnValues);
    }

    //</editor-fold>



}
