package org.jsmth.data.jdbc;

import org.apache.commons.dbcp.BasicDataSource;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.jsmth.data.dialect.Dialect;
import org.jsmth.data.dialect.DialectFactory;
import org.jsmth.jorm.jdbc.JPARowMapper;
import org.jsmth.jorm.jdbc.Table;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.PreparedStatementCallback;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.support.KeyHolder;
import org.springframework.jdbc.support.rowset.SqlRowSet;

import javax.annotation.PostConstruct;
import javax.persistence.MappedSuperclass;
import javax.sql.DataSource;
import java.net.SocketException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by mason on 16/1/4.
 */
@MappedSuperclass
public class BaseJdbcDao implements IBaseJdbcDao {

    protected NamedParameterJdbcTemplate jdbcTemplate;

    protected Logger logger = LoggerFactory.getLogger(this.getClass());
    Dialect dialect;
    DataSource dataSource;

    //<editor-fold desc="BaseJdbcDao">
    public BaseJdbcDao(DataSource dataSource) {
        this(dataSource, (Dialect) null);
    }

    public BaseJdbcDao(SmthDataSource dataSource) {
        this(dataSource, dataSource.getDialect());
    }

    public BaseJdbcDao(DataSource dataSource, String dialectString) {
        this(dataSource, (Dialect) null);
        if(StringUtils.isNotBlank(dialectString)) {
            this.dialect = DialectFactory.getDialect(dialectString);
        }else{
            if (SmthDataSource.class.isAssignableFrom(dataSource.getClass())) {
                SmthDataSource ds = (SmthDataSource) dataSource;
                this.dialect = ds.getDialect();
            } else {
                this.dialect = DialectFactory.getDialect((BasicDataSource) dataSource);
            }
        }
    }

    BaseJdbcDao(DataSource dataSource, Dialect dialect) {
        this.dataSource=dataSource;
        jdbcTemplate = new NamedParameterJdbcTemplate(dataSource);
        if (dialect == null) {
            if (SmthDataSource.class.isAssignableFrom(dataSource.getClass())) {
                SmthDataSource ds = (SmthDataSource) dataSource;
                this.dialect = ds.getDialect();
            } else {
                this.dialect = DialectFactory.getDialect((BasicDataSource) dataSource);
            }
        } else {
            this.dialect = dialect;
        }
    }
    //</editor-fold>

    @PostConstruct
    public void init() {
        Validate.notNull(dataSource, "datasource must be set!");
        Validate.notNull(jdbcTemplate, "entity dao must be set!");
        Validate.notNull(dialect, "entity dao must be set!");
    }

    //<editor-fold desc="executeNoQuery">


    public int executeNoQuery(String sql) throws DataAccessException  {
        return executeNoQuery(sql, new HashMap<String, Object>());
    }

    public int executeNoQuery(String sql, Map<String, ?> paramMap)  throws DataAccessException {
        return update(sql, paramMap);
    }

    //</editor-fold>
    //<editor-fold desc="executeNoQuery">
    public int[] batchExecuteNoQuery(String sql, Map<String, ?>[] paramMaps) throws DataAccessException  {
        return batchUpdate(sql, paramMaps);
    }

    public int[] batchExecuteNoQuery(String sql, SqlParameterSource[] paramMaps)  throws DataAccessException {
        outSQL(sql);
        return batchUpdate(sql, paramMaps);
    }
    //</editor-fold>


    //<editor-fold desc="delegate method">

    //<editor-fold desc="execute">
    public <T> T execute(String sql, SqlParameterSource paramSource, PreparedStatementCallback<T> action) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().execute(sql, paramSource, action);
    }

    public <T> T execute(String sql, Map<String, ?> paramMap, PreparedStatementCallback<T> action) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().execute(sql, paramMap, action);
    }

    public <T> T execute(String sql, PreparedStatementCallback<T> action) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().execute(sql, action);
    }
    //</editor-fold>

    //<editor-fold desc="query">
    public <T> T query(String sql, SqlParameterSource paramSource, ResultSetExtractor<T> rse) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, paramSource, rse);
    }

    public <T> T query(String sql, Map<String, ?> paramMap, ResultSetExtractor<T> rse) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, paramMap, rse);
    }

    public <T> T query(String sql, ResultSetExtractor<T> rse) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, rse);
    }

    public void query(String sql, SqlParameterSource paramSource, RowCallbackHandler rch) throws DataAccessException {
        outSQL(sql);
        getJdbcTemplate().query(sql, paramSource, rch);
    }

    public void query(String sql, Map<String, ?> paramMap, RowCallbackHandler rch) throws DataAccessException {
        outSQL(sql);
        getJdbcTemplate().query(sql, paramMap, rch);
    }

    public void query(String sql, RowCallbackHandler rch) throws DataAccessException {
        outSQL(sql);
        getJdbcTemplate().query(sql, rch);
    }

    public <T> List<T> query(String sql, SqlParameterSource paramSource, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, paramSource, rowMapper);
    }

    public <T> List<T> query(String sql, Map<String, ?> paramMap, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, paramMap, rowMapper);
    }

    public <T> List<T> query(String sql, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().query(sql, rowMapper);
    }
    public <T> T query(String sql,Object[] objects, org.springframework.jdbc.core.ResultSetExtractor<T> resultSetExtractor) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().getJdbcOperations().query(sql,objects,resultSetExtractor);
    }
    public <T> List<T> query(String sql,Object[] objects, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        try {
            return getJdbcTemplate().getJdbcOperations().query(sql, objects, rowMapper);
        }catch (Exception exception){
            exception.printStackTrace();
            if(SocketException.class.equals(exception.getClass())){
                return getJdbcTemplate().getJdbcOperations().query(sql, objects, rowMapper);
            }
            throw exception;
        }
    }
    //</editor-fold>

    //<editor-fold desc="queryForObject">
    public <T> T queryForObject(String sql, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().getJdbcOperations().queryForObject(sql, rowMapper);
    }
    public <T> T queryForObject(String sql,Object[] params, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().getJdbcOperations().queryForObject(sql,params, rowMapper);
    }

    public <T> T queryForObject(String sql, SqlParameterSource paramSource, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramSource, rowMapper);
    }

    public <T> T queryForObject(String sql, Map<String, ?> paramMap, RowMapper<T> rowMapper) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramMap, rowMapper);
    }

    public <T> T queryForObject(String sql, SqlParameterSource paramSource, Class<T> requiredType) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramSource, requiredType);
    }

    public <T> T queryForObject(String sql, Map<String, ?> paramMap, Class<T> requiredType) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramMap, requiredType);
    }
    //</editor-fold>

    //<editor-fold desc="queryForMap">
    public Map<String, Object> queryForMap(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForMap(sql, paramSource);
    }

    public Map<String, Object> queryForMap(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForMap(sql, paramMap);
    }
    @Deprecated
    public long queryForLong(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramSource,Long.class);
    }

    @Deprecated
    public long queryForLong(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramMap,Long.class);
    }

    @Deprecated
    public int queryForInt(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramSource,Integer.class);
    }

    @Deprecated
    public int queryForInt(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForObject(sql, paramMap,Integer.class);
    }
    //</editor-fold>

    //<editor-fold desc="queryForList">
    public <T> List<T> queryForList(String sql, SqlParameterSource paramSource, Class<T> elementType) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForList(sql, paramSource, elementType);
    }

    public <T> List<T> queryForList(String sql, Map<String, ?> paramMap, Class<T> elementType) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForList(sql, paramMap, elementType);
    }

    public List<Map<String, Object>> queryForList(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForList(sql, paramSource);
    }

    public List<Map<String, Object>> queryForList(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForList(sql, paramMap);
    }
    //</editor-fold>

    //<editor-fold desc="QueryForRowSet">
    public SqlRowSet queryForRowSet(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForRowSet(sql, paramSource);
    }

    public SqlRowSet queryForRowSet(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().queryForRowSet(sql, paramMap);
    }
    //</editor-fold>

    //<editor-fold desc="Update">
    public int update(String sql) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().getJdbcOperations().update(sql);
    }
    public int update(String sql,Object... params) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().getJdbcOperations().update(sql,params);
    }

    public int update(String sql, SqlParameterSource paramSource) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().update(sql, paramSource);
    }

    public int update(String sql, Map<String, ?> paramMap) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().update(sql, paramMap);
    }

    public int update(String sql, SqlParameterSource paramSource, KeyHolder generatedKeyHolder) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().update(sql, paramSource, generatedKeyHolder);
    }

    public int update(String sql, SqlParameterSource paramSource, KeyHolder generatedKeyHolder, String[] keyColumnNames) throws DataAccessException {
        outSQL(sql);
        return getJdbcTemplate().update(sql, paramSource, generatedKeyHolder, keyColumnNames);
    }
    //</editor-fold>

    //<editor-fold desc="batchUpdate">
    public int[] batchUpdate(String sql, Map<String, ?>[] batchValues) {
        outSQL(sql);
        return getJdbcTemplate().batchUpdate(sql, batchValues);
    }

    public int[] batchUpdate(String sql, SqlParameterSource[] batchArgs) {
        outSQL(sql);
        return getJdbcTemplate().batchUpdate(sql, batchArgs);
    }
    //</editor-fold>
    //</editor-fold>


    protected void outSQL(String value){
        outLog("default",value);
//        System.out.println("SQL:"+value);
    }
    protected void outLog(String tag,String value){
        logger.info(tag+":"+value);
        //System.out.println(tag+":"+value);
    }


    public Dialect getDialect() {
        return dialect;
    }

    public void setDialect(Dialect dialect) {
        this.dialect = dialect;
    }

    public DataSource getDataSource() {
        return dataSource;
    }

    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public NamedParameterJdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    public void setJdbcTemplate(NamedParameterJdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }


    //<editor-fold desc="execute">
    public List<Map<String,Object>> queryForListMap(String sql){
        return this.getJdbcTemplate().getJdbcOperations().queryForList(sql);
    }
    public <T> T queryForObject(Class<T> requestType, String sql, Object... sqlParams) {
        logger.debug("queryForObject " + sql);
        return this.jdbcTemplate.getJdbcOperations().queryForObject(sql, sqlParams, requestType);
    }

    public <T> List<T> queryForList(Class<T> requestType, String sql, Object... sqlParams) {
        logger.debug("queryForList " + sql);
        //return jdbcTemplate.query(sql, new JpaBeanPropertyRowMapper<T>(entityClasss),sqlParams);
        return this.jdbcTemplate.getJdbcOperations().queryForList(sql, sqlParams, requestType);
    }

    public Map<String, Object> queryForMap(String sql, Object... sqlParams) {
        logger.debug("queryForMap " + sql);
        return this.jdbcTemplate.getJdbcOperations().queryForMap(sql, sqlParams);
    }
    public List<Map<String, Object>> queryForListMap(String sql, Object... sqlParams) {
        logger.debug("queryForListMap " + sql);
        return this.jdbcTemplate.getJdbcOperations().queryForList(sql, sqlParams);
    }

    public <T> List<T> queryForEntityList(Class<T> entityType, String sql, Object... sqlParams) {
        logger.debug("queryForEntityList " + sql);
        return jdbcTemplate.getJdbcOperations().query(sql, new JPARowMapper<T>(entityType), sqlParams);
    }
    public <T> int execUpdate(String sql, Object... sqlParams) {
        logger.debug("Execute Update " + sql);
        return jdbcTemplate.getJdbcOperations().update(sql, sqlParams);
    }
    //</editor-fold>

    public String buildQuerySql(Class entityClass, String select,String where){
        Table table = Table.getTable(entityClass);
        StringBuilder sql=new StringBuilder();
        sql.append("select ");
        if(StringUtils.isNotBlank(select)) {
            sql.append(select);
        }
        sql.append(" from ");
        sql.append(table.getTableName());
        if(StringUtils.isNotBlank(where)) {
            sql.append(" where ");
            sql.append(where);
        }
        return sql.toString();
    }
    public String buildDeleteSql(Class entityClass, String where){
        Table table = Table.getTable(entityClass);
        StringBuilder sql=new StringBuilder();
        sql.append("delete ");
        sql.append(" from ");
        sql.append(table.getTableName());
        if(StringUtils.isNotBlank(where)) {
            sql.append(" where ");
            sql.append(where);
        }
        return sql.toString();
    }

    @Override
    public List<Map<String,Object>> findMaps(Class entityClass, String where, Object... params) {
        return findMaps(entityClass,"*",where,params);
    }

    @Override
    public List<Map<String,Object>> findMaps(Class entityClass, String select, String where, Object... params) {
        Table table = Table.getTable(entityClass);
        StringBuilder sql=new StringBuilder();
        sql.append("select ");
        sql.append(select);
        sql.append(" from ");
        sql.append(table.getTableName());
        if(StringUtils.isNotBlank(where)) {
            sql.append(" where ");
            sql.append(where);
        }
        return jdbcTemplate.getJdbcOperations().queryForList(sql.toString(),params);
    }

    @Override
    public Map getMap(Class entityClass, String where, Object... params) {
        return getMap(entityClass,"*",where,params);
    }

    @Override
    public Map<String,Object> getMap(Class entityClass, String select, String where, Object... params) {
        List<Map<String,Object>> maps = findMaps(entityClass, select, where, params);
        if(maps==null || maps.size()==0){
            return new HashMap();
        }
        return maps.get(0);
    }
}
