package org.jsmth.data.dao;

import org.jsmth.data.jdbc.ITableJdbcDao;
import org.jsmth.data.jdbc.Query;
import org.jsmth.data.jdbc.TableJdbcDao;
import org.jsmth.data.sql.EntityQuery;
import org.jsmth.data.sql.wrap.WhereWrap;
import org.jsmth.domain.Identifier;
import org.jsmth.jorm.jdbc.EntityEventCallback;
import org.jsmth.jorm.jdbc.Event;
import org.jsmth.jorm.jdbc.Table;
import org.jsmth.page.CommonPage;
import org.jsmth.page.Page;
import org.jsmth.util.BeanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.persistence.MappedSuperclass;
import java.io.Serializable;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Created by mason on 15/12/26.
 */
@MappedSuperclass
public class BaseEntityDao<MODEL> implements IBaseEntityDao<MODEL>, EntityEventCallback {
    protected Logger log = LoggerFactory.getLogger(this.getClass());
    protected EntityEventCallback entityEventCallback;
    protected ITableJdbcDao tableJdbcDao;
    protected Class<MODEL> entityClass;

    //<editor-fold desc="Description">
    public BaseEntityDao(Class<MODEL> entityClass) {
        this.entityClass = entityClass;
    }

    public BaseEntityDao(Class<MODEL> entityClass, EntityEventCallback entityEventCallback) {
        this.entityClass = entityClass;
        this.entityEventCallback = entityEventCallback;
    }

    public BaseEntityDao(Class<MODEL> entityClass, TableJdbcDao baseJdbcDao) {
        this.entityClass = entityClass;
        setTableJdbcDao(baseJdbcDao);
    }
    //</editor-fold>

    //<editor-fold desc="Description">
    public List<MODEL> findAll() {
        return getTableJdbcDao().queryAll(entityClass);
    }

    public Page<MODEL> findAll(int pageNumber, int pageSize) {
        return getTableJdbcDao().queryAllForPage(entityClass, pageNumber, pageSize);
    }


    public Page<MODEL> findPageModels(WhereWrap wrap, int pageNumber, int pageSize, boolean totalRecord) {
        return getTableJdbcDao().queryPage(entityClass, wrap, pageNumber, pageSize, totalRecord);
    }


    public Page<MODEL> findPageModels(EntityQuery query) {
        return getTableJdbcDao().queryPage(entityClass, query);
    }

    @Override
    public Page<MODEL> pageModels(int pageNumber, int pageSize, boolean totalRecord, String where, Object... params) {
        String sql =this.getTableJdbcDao().buildQuerySql(entityClass, "count(*)", where);
        Integer count = this.getTableJdbcDao().queryForObject(Integer.class, sql, params);
        sql = this.getTableJdbcDao().buildQuerySql(entityClass, "*", where);
        CommonPage<MODEL> page=new CommonPage<>();
        page.setPageNumber(pageNumber);
        page.setPageSize(pageSize);
        page.setTotalItemsCount(count);
        page.setItems(getTableJdbcDao().queryForEntityList(entityClass, sql,params));
        return page;
    }


    public List<MODEL> findModels(WhereWrap wrap) {
        return getTableJdbcDao().query(entityClass, wrap);
    }

    public List<MODEL> findModels(EntityQuery query) {
        return getTableJdbcDao().query(entityClass, query);
    }

    public <K> List<K> findColumns(Class<K> kClass, WhereWrap wrap) {
        return getTableJdbcDao().query(kClass, wrap);
    }


    public <K> List<K> findColumns(Class<K> kClass, EntityQuery query) {
        return getTableJdbcDao().queryColumn(kClass, query);
    }

    @Override
    public <K> List<K> findColumns(Class<K> kClass, String columnName, String where, Object... params) {
        return null;
    }

    @Override
    public <K> Page<K> pageIds(Class<K> kClass,int pageNumber, int pageSize, boolean totalRecord, String where, Object... params) {
        return pageColumns(kClass,pageNumber, pageSize,totalRecord, "id",where,params);
    }

    @Override
    public Page<MODEL> pageModels(int pageNumber, int pageSize, boolean totalRecord, Query query) {
        Table table = Table.getTable(entityClass);
        List sqlParas = new LinkedList<>();
        String sql = query.buildWhere(table.getTableName(), sqlParas);
        return pageModels(pageNumber, pageSize, totalRecord, sql, sqlParas.toArray());
    }

    @Override
    public <K> Page<K> pageColumns(Class<K> kClass, int pageNumber, int pageSize, boolean totalRecord, String select, String where, Object... params) {
        return null;
    }

    public <K> Page<K> pageColumns(Class<K> kClass, WhereWrap wrap, int pageNumber, int pageSize, boolean totalRecord) {
        return getTableJdbcDao().queryPage(kClass, wrap, pageNumber, pageSize, totalRecord);
    }


    public <K> Page<K> pageColumns(Class<K> kClass, EntityQuery query) {
        return getTableJdbcDao().queryPageColumn(kClass, query);
    }

    public MODEL getModel(WhereWrap where) {
        return getTableJdbcDao().queryForObject(entityClass, where);
    }

    public MODEL getModel(EntityQuery query) {
        return getTableJdbcDao().queryForObject(entityClass, query);
    }


    /**
     * 根据条件，查询得到一组model
     *
     * @param where ff
     * @param params ff
     * @return 返回信息
     */
    public List<MODEL> findModels(String where, Object... params) {
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW(where,params);
        return getTableJdbcDao().query(entityClass, whereWrap);
    }



    public Page<MODEL> pageFindModels(String where, int pageNumber, int pageSize, boolean isGetTotal, Object... params) {
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW(where,params);
        return getTableJdbcDao().queryPage(entityClass, whereWrap, pageNumber, pageSize);
    }


    public <T, E> List<E> groupColumn(Class<E> columnClass, String fieldName, String where, Object... params) {
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW(where,params);
        return getTableJdbcDao().queryColumn(entityClass, columnClass, fieldName, whereWrap);
    }

    public <T> List<T> findColumn(Class<T> columnClass, String fieldName, String where, Object... params) {
        WhereWrap whereWrap=new WhereWrap();
        whereWrap.placeholderW(where,params);
        return getTableJdbcDao().queryColumn(entityClass, columnClass, fieldName, whereWrap);
    }

    public <T> Page<T> pageFindColumn(Class<T> columnClass, String fieldName, String where, int pageNumber, int pageSize, Object... params) {
        return pageColumns(columnClass,pageNumber,pageSize,true,fieldName,where,params);
    }
    public <T> Page<T> pageFindColumn(Class<T> columnClass, String fieldName,  int pageNumber, int pageSize, Query query){
        Table table = Table.getTable(entityClass);
        List sqlParas = new LinkedList<>();
        String sql = query.buildWhere(table.getTableName(), sqlParas);
        return pageFindColumn(columnClass,fieldName,sql,pageNumber,pageSize,sqlParas);
    }

    @Override
    public void rebuildSchema(Class... entityClass) {
        tableJdbcDao.rebuildSchema(entityClass);
    }


    @Override
    public List<MODEL> query(WhereWrap where) {
        return tableJdbcDao.query(entityClass, where);
    }

    @Override
    public MODEL queryForObject(WhereWrap where) {
        return tableJdbcDao.queryForObject(entityClass, where);
    }

    @Override
    public List<Map<String, Object>> queryForList(WhereWrap where) {
        return tableJdbcDao.queryForList(entityClass, where);
    }



    @Override
    public void batchInvokeEvent(Collection targets, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.batchInvokeEvent(targets, event);
        }
    }

    @Override
    public void invokeEvent(Object target, Event event) {
        if (entityEventCallback != null) {
            entityEventCallback.invokeEvent(target, event);
        }
    }

    public final ITableJdbcDao getTableJdbcDao() {
        return tableJdbcDao;
    }

    @Override
    public List<Map<String, Object>> queryForListMap(String sql) {
        return getTableJdbcDao().queryForListMap(sql);
    }



    public final void setTableJdbcDao(ITableJdbcDao tableJdbcDao) {
        this.tableJdbcDao = tableJdbcDao;
        this.tableJdbcDao.setEntityEventCallback(this);
    }

    public Class<MODEL> getEntityClass() {
        return entityClass;
    }

    @Override
    public Class getKeyClass() {
        MODEL instance = BeanUtils.createInstance(getEntityClass());
        Identifier identifier = (Identifier) instance;
        return identifier.getKeyClass();
    }

    public void setEntityClass(Class<MODEL> entityClass) {
        this.entityClass = entityClass;
    }

    public EntityEventCallback getEntityEventCallback() {
        return entityEventCallback;
    }

    public void setEntityEventCallback(EntityEventCallback entityEventCallback) {
        this.entityEventCallback = entityEventCallback;
    }
    //</editor-fold>



}
