package org.jsmth.data.code.sqlbuilder.select;

import org.jsmth.data.code.sqlbuilder.Context;
import org.jsmth.data.code.sqlbuilder.Database;
import org.jsmth.data.code.sqlbuilder.TerminalExpression;
import org.jsmth.data.dialect.Dialect;
import org.jsmth.data.dialect.HSQLDialect;
import org.jsmth.data.dialect.OracleDialect;

import java.sql.SQLException;
import java.util.List;

public class Limit
{

	private final Context context;

	public Limit(Context context, int start, int size) {
		this.context = limit(context, start, size);
	}


	private Context limit(Context context, int start, int size) {
		return new LimiterFactory().create(context.getDialect()).limit(context, start, size);
	}

	@Override
	public String toString() {
		return context.toString();
	}
}

interface Limiter {
	Context limit(Context context, int start, int size);
}

class HSQLDBLimiter implements Limiter {

	@Override
	public Context limit(Context context, int start, int size) {
		context.appendLine("LIMIT ?");
		context.addParameters(size);
		context.appendLine("OFFSET ?");
		context.addParameters(start);
		return context;
	}

}

class OracleLimiter implements Limiter {

	@Override
	public Context limit(Context context, int start, int size) {
		Context c = new Context(context);
		c.appendLine("SELECT");
		c.appendLine("data.*");
		c.appendLine("FROM");
		c.appendLine("(");
		c.appendLine("SELECT");
		c.appendLine("ord_data.*,");
		c.appendLine("rownum AS rnum");
		c.appendLine("FROM");
		c.appendLine("(");
		c.appendLine(context.toString());
		c.appendLine(")");
		c.appendLine("ord_data");
		c.appendLine(")");
		c.appendLine("data");
		c.appendLine("WHERE");
		c.appendLine("rnum BETWEEN ? AND ?");
		c.addParameters(start);
		c.addParameters(start + size);
		return c;
	}

}

class DefaultLimiter implements Limiter {
	@Override
	public Context limit(Context context, int start, int size) {
		return context;
	}
}

class LimiterFactory {
	Limiter create(Dialect dialect) {
		if(HSQLDialect.class.isInstance(dialect)){
			return new HSQLDBLimiter();
		}
		if(OracleDialect.class.isInstance(dialect)){
			return new OracleLimiter();
		}
		return new DefaultLimiter();
	}
}