/*
 * Copyright (c) 2011-2021, baomidou (jobob@qq.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package cn.net.vidyo.framework.builder.converts;


import cn.net.vidyo.framework.builder.config.GlobalConfig;
import cn.net.vidyo.framework.builder.domain.DateType;
import cn.net.vidyo.framework.builder.domain.DbColumnType;
import cn.net.vidyo.framework.builder.domain.IColumnType;
import cn.net.vidyo.framework.builder.domain.ITypeConvert;

import static cn.net.vidyo.framework.builder.converts.TypeConverts.contains;
import static cn.net.vidyo.framework.builder.converts.TypeConverts.containsAny;
import static cn.net.vidyo.framework.builder.domain.DbColumnType.*;

/**
 * SQLServer 字段类型转换
 */
public class SqlServerTypeConvert implements ITypeConvert {

    public static final SqlServerTypeConvert INSTANCE = new SqlServerTypeConvert();


    @Override
    public IColumnType processTypeConvert(DateType dateType, String fieldType) {
        return TypeConverts.use(fieldType)
            .test(containsAny("char", "xml", "text").then(STRING))
            .test(contains("bigint").then(LONG))
            .test(contains("int").then(INTEGER))
            .test(containsAny("date", "time").then(t -> toDateType(dateType, t)))
            .test(contains("bit").then(BOOLEAN))
            .test(containsAny("decimal", "numeric").then(DOUBLE))
            .test(contains("money").then(BIG_DECIMAL))
            .test(containsAny("binary", "image").then(BYTE_ARRAY))
            .test(containsAny("float", "real").then(FLOAT))
            .or(STRING);
    }

    /**
     * 转换为日期类型
     *
     * @param dateType 配置信息
     * @param fieldType   类型
     * @return 返回对应的列类型
     */
    public static IColumnType toDateType(DateType dateType, String fieldType) {
        switch (dateType) {
            case SQL_PACK:
                switch (fieldType) {
                    case "date":
                        return DATE_SQL;
                    case "time":
                        return TIME;
                    default:
                        return TIMESTAMP;
                }
            case TIME_PACK:
                switch (fieldType) {
                    case "date":
                        return LOCAL_DATE;
                    case "time":
                        return LOCAL_TIME;
                    default:
                        return LOCAL_DATE_TIME;
                }
            default:
                return DATE;
        }
    }
}
