/*
 * Copyright (c) 2011-2021, baomidou (jobob@qq.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package cn.net.vidyo.framework.builder.converts;


import cn.net.vidyo.framework.builder.config.GlobalConfig;
import cn.net.vidyo.framework.builder.domain.DateType;
import cn.net.vidyo.framework.builder.domain.DbColumnType;
import cn.net.vidyo.framework.builder.domain.IColumnType;
import cn.net.vidyo.framework.builder.domain.ITypeConvert;

import static cn.net.vidyo.framework.builder.converts.TypeConverts.contains;
import static cn.net.vidyo.framework.builder.converts.TypeConverts.containsAny;
import static cn.net.vidyo.framework.builder.domain.DbColumnType.*;

/**
 * MYSQL 数据库字段类型转换
 *
 *
 *
 */
public class FirebirdTypeConvert implements ITypeConvert {
    public static final FirebirdTypeConvert INSTANCE = new FirebirdTypeConvert();


    @Override
    public IColumnType processTypeConvert(DateType dateType, String fieldType) {
        return TypeConverts.use(fieldType)
            .test(containsAny("cstring", "text").then(STRING))
            .test(contains("short").then(SHORT))
            .test(contains("long").then(LONG))
            .test(contains("float").then(FLOAT))
            .test(contains("double").then(DOUBLE))
            .test(contains("blob").then(BLOB))
            .test(contains("int64").then(LONG))
            .test(containsAny("date", "time", "year").then(t -> toDateType(dateType, t)))
            .or(STRING);
    }

    /**
     * 转换为日期类型
     *
     * @param dateType 配置信息
     * @param type   类型
     * @return 返回对应的列类型
     */
    public static IColumnType toDateType(DateType dateType,String type) {
        switch (dateType) {
            case ONLY_DATE:
                return DbColumnType.DATE;
            case SQL_PACK:
                switch (type) {
                    case "date":
                    case "year":
                        return DbColumnType.DATE_SQL;
                    case "time":
                        return DbColumnType.TIME;
                    default:
                        return DbColumnType.TIMESTAMP;
                }
            case TIME_PACK:
                switch (type) {
                    case "date":
                        return DbColumnType.LOCAL_DATE;
                    case "time":
                        return DbColumnType.LOCAL_TIME;
                    case "year":
                        return DbColumnType.YEAR;
                    default:
                        return DbColumnType.LOCAL_DATE_TIME;
                }
        }
        return STRING;
    }

}
