/*
 * Copyright (c) 2011-2021, baomidou (jobob@qq.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package cn.net.vidyo.framework.builder.query;

import cn.net.vidyo.framework.builder.config.GlobalConfig;
import cn.net.vidyo.framework.builder.config.StrategyConfig;
import cn.net.vidyo.framework.builder.domain.IColumnType;
import cn.net.vidyo.framework.builder.meta.ColumnSchema;
import cn.net.vidyo.framework.builder.meta.TableSchema;
import cn.net.vidyo.framework.builder.querys.DbQueryDecorator;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 后面切换到元数据获取表与字段会移除这里
 *
 * @author nieqiurong 2021/1/6.
 * @since 3.5.0
 */
public class DefaultDatabaseQuery extends AbstractDatabaseQuery {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDatabaseQuery.class);

    private final StrategyConfig strategyConfig;

    private final GlobalConfig globalConfig;

    private final DbQueryDecorator dbQuery;

    private final DatabaseMetaDataWrapper databaseMetaDataWrapper;
    ConnectManager connectManager;

    public DefaultDatabaseQuery(GlobalConfig globalConfig) {
        super(globalConfig.getDataSourceConfig());
        this.strategyConfig = globalConfig.getStrategyConfig();
        this.dbQuery = new DbQueryDecorator(dataSourceConfig, strategyConfig);
        this.globalConfig = globalConfig;
        connectManager =new ConnectManager(dataSourceConfig);
        this.databaseMetaDataWrapper = new DatabaseMetaDataWrapper(connectManager.getConnection());
    }



    @Override
    public List<TableSchema> queryTables() {
        boolean isInclude = strategyConfig.getInclude().size() > 0;
        boolean isExclude = strategyConfig.getExclude().size() > 0;
        //所有的表信息
        List<TableSchema> tableList = new ArrayList<>();

        //需要反向生成或排除的表信息
        List<TableSchema> includeTableList = new ArrayList<>();
        List<TableSchema> excludeTableList = new ArrayList<>();
        try {
            connectManager.execute(dbQuery.tablesSql(),  result -> {
                String tableName = result.getStringResult(dbQuery.tableName());
                if (StringUtils.isNotBlank(tableName)) {
                    DatabaseMetaDataWrapper.Table table = databaseMetaDataWrapper.getTableInfo(tableName);
                    TableSchema tableInfo = new TableSchema();
                    tableInfo.setName(tableName);
                    // 跳过视图
                    if (!(strategyConfig.isSkipView() && table.isView())) {
                        tableInfo.setComment(table.getRemarks());
                        if (isInclude && strategyConfig.matchIncludeTable(tableName)) {
                            includeTableList.add(tableInfo);
                        } else if (isExclude && strategyConfig.matchExcludeTable(tableName)) {
                            excludeTableList.add(tableInfo);
                        }
                        tableList.add(tableInfo);
                    }
                }
            });
            if (isExclude || isInclude) {
                Map<String, String> notExistTables = new HashSet<>(isExclude ? strategyConfig.getExclude() : strategyConfig.getInclude())
                    .stream()
                    .filter(s -> !matcherRegTable(s))
                    .collect(Collectors.toMap(String::toLowerCase, s -> s, (o, n) -> n));
                // 将已经存在的表移除，获取配置中数据库不存在的表
                for (TableSchema tabInfo : tableList) {
                    if (notExistTables.isEmpty()) {
                        break;
                    }
                    //解决可能大小写不敏感的情况导致无法移除掉
                    notExistTables.remove(tabInfo.getName().toLowerCase());
                }
                if (notExistTables.size() > 0) {
                    LOGGER.warn("表[{}]在数据库中不存在！！！", String.join(",", notExistTables.values()));
                }
                // 需要反向生成的表信息
                if (isExclude) {
                    tableList.removeAll(excludeTableList);
                } else {
                    tableList.clear();
                    tableList.addAll(includeTableList);
                }
            }
            // 性能优化，只处理需执行表字段 https://github.com/baomidou/mybatis-plus/issues/219
            tableList.forEach(this::convertTableFields);
            return tableList;
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            // 数据库操作完成,释放连接对象
            connectManager.closeConnection();
        }
    }

    /**
     * 过滤正则
     */
    private static final Pattern REGX = Pattern.compile("[~!/@#$%^&*()+\\\\\\[\\]|{};:'\",<.>?]+");
    /**
     * 判断表名是否为正则表名(这表名规范比较随意,只能尽量匹配上特殊符号)
     *
     * @param tableName 表名
     * @return 是否正则
     * @since 3.5.0
     */
    public static boolean matcherRegTable(String tableName) {
        return REGX.matcher(tableName).find();
    }

    private void convertTableFields( TableSchema tableInfo) {
        String tableName = tableInfo.getName();
        try {
            final Map<String, DatabaseMetaDataWrapper.ColumnsInfo> columnsMetaInfoMap = new HashMap<>();
            //TODO 增加元数据信息获取,后面查询表字段要改成这个.
            Map<String, DatabaseMetaDataWrapper.ColumnsInfo> columnsInfoMap =
                databaseMetaDataWrapper.getColumnsInfo(tableName);
            if (columnsInfoMap != null && !columnsInfoMap.isEmpty()) {
                columnsMetaInfoMap.putAll(columnsInfoMap);
            }
            String tableFieldsSql = dbQuery.tableFieldsSql(tableName);
//            Entity entity = strategyConfig.entity();
            connectManager.execute(tableFieldsSql, result -> {
                String columnName = result.getStringResult(dbQuery.fieldName());
                ColumnSchema field = new ColumnSchema();
                field.setName(columnName);
                // 避免多重主键设置，目前只取第一个找到ID，并放到list中的索引为0的位置
                DatabaseMetaDataWrapper.ColumnsInfo columnsInfo = columnsMetaInfoMap.get(columnName.toLowerCase());
                // 处理ID
                if (columnsInfo.isPrimaryKey()) {
                    field.setPrimaryKey(columnsInfo.isAutoIncrement());
                    tableInfo.setIdColumn(field);
//                    if (field.isKeyIdentityFlag() && entity.getIdType() != null) {
//                        LOGGER.warn("当前表[{}]的主键为自增主键，会导致全局主键的ID类型设置失效!", tableName);
//                    }
                }
                field.setName(columnName);
                field.setDbColumnTypeString(result.getStringResult(dbQuery.fieldType()));
                String typeName = field.getDbColumnTypeString();
                int pos = typeName.indexOf("(");
                if(pos>0){
                    typeName=typeName.substring(0,pos);
                }
                field.setDataTypeName(typeName);
                field.setComment(columnsInfo.getRemarks());
                field.setCustomMap(dbQuery.getCustomFields(result.getResultSet()));
//                String propertyName = entity.getNameConvert().propertyNameConvert(field);
                IColumnType columnType = dataSourceConfig.getTypeConvert().processTypeConvert(globalConfig, field);
                field.setPropertyName(columnsInfo.getName());
                field.setColumnType(columnType);
                if(columnName.equalsIgnoreCase("status")){
                    System.out.println("Table " +tableInfo.getName() +" exist status");
                    tableInfo.setIncludeStatus(true);
                }
//                field.setPropertyName(propertyName, columnType);
//                field.setMetaInfo(new TableField.MetaInfo(columnsInfo));
                tableInfo.addColumn(field);
            });
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
//        tableInfo.processTable();
    }

}
