package cn.net.vidyo.framework.data.jpa.dao;

import cn.net.vidyo.framework.common.util.ObjectUtil;
import cn.net.vidyo.framework.common.util.StringUtils;
import cn.net.vidyo.framework.data.jpa.dao.sql.DeleteWrapper;
import cn.net.vidyo.framework.data.jpa.dao.sql.UpdateWrapper;
import cn.net.vidyo.framework.data.jpa.dao.sql.WhereWrapper;
import cn.net.vidyo.framework.data.jpa.dao.sql.QueryWrapper;
import cn.net.vidyo.framework.data.jpa.dao.support.DefaultEntityEventCallback;
import org.hibernate.SQLQuery;
import org.hibernate.SessionFactory;
import org.hibernate.internal.SessionFactoryImpl;
import org.hibernate.persister.entity.SingleTableEntityPersister;
import org.hibernate.transform.Transformers;
import org.springframework.data.domain.*;
import org.springframework.data.jpa.repository.support.JpaEntityInformation;
import org.springframework.data.jpa.repository.support.SimpleJpaRepository;

import javax.persistence.*;
import java.io.Serializable;
import java.util.*;

public class CommonJpaRepositoryImpl<T, ID extends Serializable> extends SimpleJpaRepository<T, ID> implements EntityEventCallback, CommonJpaRepository<T, ID> {
    static int BATCH_SIZE = 10000;
    private final EntityManager entityManager;
    JpaEntityInformation<T, ?> entityInformation;
    Class<T> entityClass;
    EntityEventCallback defaultEntityEventCallback;

    @Override
    public Class<T> getEntityClass() {
        return entityClass;
    }
    public CommonJpaRepositoryImpl(JpaEntityInformation<T, ?> entityInformation, EntityManager entityManager) {
        super(entityInformation, entityManager);
        this.entityManager = entityManager;
        this.entityInformation = entityInformation;
        this.entityClass = entityInformation.getJavaType();
        defaultEntityEventCallback = new DefaultEntityEventCallback();
    }

    //<editor-fold desc="Extent">
    //<editor-fold desc="update">
    //@Transactional(readOnly = false)
    @Override
    public int updateStatusById(ID id, Object value) {
        return updateColumnById(id, "status", value);
    }

    //@Transactional(readOnly = false)
    @Override
    public int updateHiddenById(ID id, Object value) {
        return updateColumnById(id, "hidden", value);
    }

    //@Transactional(readOnly = false)
    @Override
    public int updateColumnById(ID id, String fieldName, Object value) {
        return updateColumn(fieldName, value, new QueryWrapper().inId(id));
    }

    //@Transactional(readOnly = false)
    public int increaseColumnValueById(ID id, String fieldName, int delta) {
        return increaseColumn(fieldName, delta, "id=?", id);
    }

    //</editor-fold>
    //<editor-fold desc="delete">
    @Override
    public int deleteByIds(Iterable<ID> ids) {
        if(ids==null) return 0;
        WhereWrapper wrapper=new WhereWrapper();
        wrapper.in("id",ids);
        return delete(wrapper);
    }

    public int deleteByIds(ID... ids) {
        if(ids==null) return 0;
        WhereWrapper wrapper=new WhereWrapper();
        wrapper.eq(ids.length==1,"id",ids[0]);
        wrapper.in(ids.length>0,"id",ids);
        return delete(wrapper);
    }

    //@Transactional(readOnly = false)
    @Override
    public void deleteAll() {
        deleteByWhere(new QueryWrapper());
    }

    //@Transactional(readOnly = false)
    @Override
    public void deleteAll(Iterable<? extends T> entities) {
        List<ID> ids = new ArrayList<>();
        for (T entity : entities) {
            ID id = getIdByEntity(entity);
            ids.add(id);
        }
        deleteByWhere(new QueryWrapper().in("id", ids));
    }

    //</editor-fold>
    //<editor-fold desc="Query">
    //<editor-fold desc="get entity">
    @Override
    public T getById(ID id) {
        Optional<T> optional = findById(id);
        if (!optional.isPresent()) {
            return null;
        }
        T t = optional.get();
        invokeEvent(t, Event.PostLoad);
        return t;
    }

    public ID getIdByEntity(T t) {
        return (ID) ObjectUtil.getFieldValueByFieldName(t, "id");
    }

    @Override
    public T getByEntityId(T t) {
        ID id = getIdByEntity(t);
        return getById(id);
    }

    @Override
    public int truncateParmeryKey(Class entityClass) {
        StringBuilder sql = new StringBuilder();
        sql.append("TRUNCATE TABLE ");
        sql.append(getTableName(entityClass));
        return execUpdate(sql.toString());
    }

    public int dropTable(Class entityClass) {
        StringBuilder sql = new StringBuilder();
        sql.append("DROP TABLE ");
        sql.append(getTableName(entityClass));
        return execUpdate(sql.toString());
    }


    //</editor-fold>
    //<editor-fold desc="get column">
    @Override
    public Object getColumnByWhere(String fieldName,WhereWrapper wrapper) {
        QueryWrapper queryWrapper=new QueryWrapper(getTableName(), wrapper);
        queryWrapper.setSelectSql(fieldName);
        Map map = getMap(queryWrapper);
        if(map!=null){
            if(map.containsKey(fieldName)){
                return map.get(fieldName);
            }
        }
        return null;
    }
    @Override
    public Object getColumnById(ID id, String fieldName) {
        WhereWrapper whereWrapper=new WhereWrapper("id",id);
        return getColumnByWhere(fieldName, whereWrapper);
    }

    public String getStringColumnById(ID id, String fieldName) {
        Object value = getColumnById(id, fieldName);
        if (value == null) {
            return "";
        }
        return value.toString();
    }

    @Override
    public String getNameById(ID id) {
        return getStringColumnById(id, "name");
    }

    public String getIdKeyById(ID id) {
        return getStringColumnById(id, "idkey");
    }

    public String getCodeById(ID id) {
        return getStringColumnById(id, "code");
    }

    //</editor-fold>
    //<editor-fold desc="find list">
    @Override
    public List<T> findByIds(List<ID> ids) {
        return this.findAllById(ids);
    }

    @Override
    public List<T> findByIds(ID[] ids) {
        List list = Arrays.asList(ids);
        return findByIds(list);
    }

    public Page<T> pageAll(int pageNumber, int pageSize) {
        return pageAll(PageRequest.of(pageNumber, pageSize));
    }

    public Page<T> pageAll() {
        return pageAll(1, 100000);
    }

    public Page<T> pageAll(Pageable pageable) {
        return pageQueryModels(pageable, new QueryWrapper());
    }

    public List<T> findAll() {
        return queryModels(new QueryWrapper());
    }

    //</editor-fold>
    //</editor-fold>
    //</editor-fold>
    //<editor-fold desc="Base">
    //<editor-fold desc="update">
    //@Transactional(readOnly = false)
    public int increaseColumn(String columName, int delta, String where, Object... params) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where,params);
        wrapper.set(true,columName,columName+"+?",delta);
        return execUpdate(wrapper);
    }

    @Override
    public int increaseColumn(String columName, int delta, WhereWrapper where) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where);
        wrapper.set(true,columName,columName+"+?",delta);
        return execUpdate(wrapper);
    }

    //@Transactional(readOnly = false)
    public int updateColumn(String columName, Object value, String where, Object... params) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where,params);
        wrapper.set(columName,value);
        return execUpdate(wrapper);
    }

    @Override
    public int updateColumn(String columName, Object value, WhereWrapper where) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where);
        wrapper.set(columName,value);
        return execUpdate(wrapper);
    }

    @Override
    public int updateColumns(Map<String, Object> columnNameValues, WhereWrapper where) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where);
        wrapper.set(true,columnNameValues);
        return execUpdate(wrapper);
    }


    //@Transactional(readOnly = false)
    public int updateColumn(String columName, Object value, QueryWrapper where) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where);
        wrapper.set(columName,value);
        return execUpdate(wrapper);
    }

    //@Transactional(readOnly = false)
    public int updateColumns(Map columnNameValues, String where, Object... params) {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where,params);
        wrapper.set(true,columnNameValues);
        return execUpdate(wrapper);
    }

    //@Transactional(readOnly = false)
    public int updateColumns(Map columnNameValues, QueryWrapper where)
    {
        UpdateWrapper wrapper=new UpdateWrapper(getTableName(), where);
        wrapper.set(true,columnNameValues);
        return execUpdate(wrapper);
    }

    //@Transactional(readOnly = false)
    public <S extends T> Iterable<S> batchUpdate(Iterable<S> var1) {
        Iterator<S> iterator = var1.iterator();
        int index = 0;
        while (iterator.hasNext()) {
            S entity = iterator.next();
            invokeEvent(entity, Event.PreUpdate);
            entityManager.merge(entity);
            invokeEvent(entity, Event.PostUpdate);
            index++;
            if (index % BATCH_SIZE == 0) {
                entityManager.flush();
                entityManager.clear();
            }
        }
        if (index % BATCH_SIZE != 0) {
            entityManager.flush();
            entityManager.clear();
        }
        return var1;
    }

    //</editor-fold>
    //<editor-fold desc="insert">
    //@Transactional(readOnly = false)
    public <S extends T> S save(S entity) {
        if (this.entityInformation.isNew(entity)) {
            invokeEvent(entity, Event.PrePersist);
            this.entityManager.persist(entity);
            invokeEvent(entity, Event.PostPersist);
            return entity;
        } else {
            invokeEvent(entity, Event.PreUpdate);
            S merge = this.entityManager.merge(entity);
            invokeEvent(merge, Event.PostUpdate);
            return merge;
        }
    }

    //@Transactional(readOnly = false)
    public <S extends T> S saveAndFlush(S entity) {
        S result = this.save(entity);
        this.flush();
        return result;
    }

    //@Transactional(readOnly = false)
    public <S extends T> Iterable<S> batchSave(Iterable<S> var1) {
        Iterator<S> iterator = var1.iterator();
        int index = 0;
        while (iterator.hasNext()) {
            S entity = iterator.next();
            invokeEvent(entity, Event.PrePersist);
            entityManager.persist(entity);
            invokeEvent(entity, Event.PostPersist);
            index++;
            if (index % BATCH_SIZE == 0) {
                entityManager.flush();
                entityManager.clear();
            }
        }
        if (index % BATCH_SIZE != 0) {
            entityManager.flush();
            entityManager.clear();
        }
        return var1;
    }

    public  <S extends T> S insert(S entity){
        invokeEvent(entity, Event.PrePersist);
        this.entityManager.persist(entity);
        invokeEvent(entity, Event.PostPersist);
        return entity;
    }
    public  <S extends T> S update(S entity){
        invokeEvent(entity, Event.PreUpdate);
        S merge = this.entityManager.merge(entity);
        invokeEvent(merge, Event.PostUpdate);
        return merge;
    }
    //</editor-fold>
    //<editor-fold desc="delete">
    //@Transactional(readOnly = false)
    public int deleteByWhere(String where, Object... params) {
        DeleteWrapper wrapper=new DeleteWrapper(getTableName(),where,params);
        return execDelete(wrapper);
    }


    @Override
    public int deleteByWhere(WhereWrapper where) {
        DeleteWrapper wrapper=new DeleteWrapper(getTableName(),where);
        return execDelete(wrapper);
    }

    //@Transactional(readOnly = false)
    public int deleteByWhere(QueryWrapper where) {
        DeleteWrapper wrapper=new DeleteWrapper(getTableName(),where);
        return execDelete(wrapper);
    }

    //</editor-fold>
    //<editor-fold desc="query">

    //<editor-fold desc="query object">
    public Object getObject(String columnName, String where, Object... params){
        return getObject(columnName,new QueryWrapper(where, params));
    }
    public Object getObject(String columnName, WhereWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(columnName);
        return execObjectQuery(wrapper);
    }

    public T getModel( String where, Object... params){
        return getModel(new QueryWrapper(where, params));
    }


    public <C> C getColumn(Class<C> cClass,String select, String where, Object... params){
        QueryWrapper queryWhere = new QueryWrapper(where, params);
        queryWhere.setSelectSql(select);
        return getColumn(cClass, queryWhere);
    }
    public <C> C getColumn(Class<C> cClass,String select, WhereWrapper where){
        QueryWrapper queryWhere = new QueryWrapper(getTableName(), where);
        queryWhere.setSelectSql(select);
        return getColumn(cClass, queryWhere);
    }

    public Map getMap(String where, Object... params){
        return execMapQuery(new QueryWrapper(getTableName(), where, params));
    }

    //</editor-fold>
    //<editor-fold desc="query object list">
    public List<T> queryModels(String where, Object... params) {
        return queryModels(new WhereWrapper(where, params));
    }

    @Override
    public List<T> queryModels(WhereWrapper where) {
        QueryWrapper selectWhere = new QueryWrapper(getTableName(), where);
        return execListModelQuery(selectWhere);
    }

    public <C> List<C> queryColumns(Class<C> cClass, String select, String where, Object... params) {
        QueryWrapper selectWhere = new QueryWrapper(where, params);
        selectWhere.setSelectSql(select);
        return queryColumns(cClass, selectWhere);
    }
    public <C> List<C> queryColumns(Class<C> cClass, QueryWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(where.getSelectSql());
        return execListColumnQuery(cClass, wrapper);
    }

    public List queryObjects(String select, String where, Object... params) {
        QueryWrapper selectWhere = new QueryWrapper(where, params);
        selectWhere.setSelectSql(select);
        return queryObjects( selectWhere);
    }
    public List queryObjects( QueryWrapper where) {
        return execListObjectQuery(where);
    }

    @Override
    public List<Map> queryMaps(String select, String where, Object... params) {
        return null;
    }

    public List<Map> queryMaps(String where, Object... params){
        return queryMaps(new QueryWrapper(where, params));
    }
    public List<Map> queryMaps(QueryWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        return execListMapQuery(wrapper);
    }

    //</editor-fold>
    //<editor-fold desc="query object page">

    public Page<T> pageQueryModels(int pageNumber, int pageSize, String where, Object... params) {
        return pageQueryModels(pageNumber, pageSize, where, params);
    }

    public Page<T> pageQueryModels(int pageNumber, int pageSize, QueryWrapper where) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize);
        return pageQueryModels(pageable, where);
    }

    @Override
    public Page<T> pageQueryModels(int pageNumber, int pageSize, WhereWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setPageNumber(pageNumber);
        wrapper.setPageSize(pageSize);
        return execPageQuery(wrapper);
    }

    public Page<T> pageQueryModels(int pageNumber, int pageSize, String select, String where, Object... params) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize);
        return pageQueryModels(pageable, select, where, params);
    }

    public Page<T> pageQueryModels(Pageable pageable, String where, Object... params) {
        return pageQueryModels(pageable, "*", where, params);
    }

    public Page<T> pageQueryModels(Pageable pageable, WhereWrapper where) {
        QueryWrapper queryWrapper=new QueryWrapper(getTableName(),where);
        queryWrapper.setPageable(pageable);
        return execPageQuery(queryWrapper);
    }
    public Page<T> pageQueryModels(QueryWrapper wrapper) {
        return execPageQuery(wrapper);
    }

    public Page<T> pageQueryModels(Pageable pageable, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setSelectSql(select);
        wrapper.setPageable(pageable);
        return execPageQuery(wrapper);
    }

    @Override
    public Page<Map> pageQueryMaps(Pageable pageable, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(select);
        wrapper.setPageable(pageable);
        return execMapPageQuery(wrapper);
    }

    public Page<Map> pageQueryMaps(Pageable pageable, QueryWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(where.getSelectSql());
        wrapper.setPageable(pageable);
        return execMapPageQuery(wrapper);
    }
    public Page<Map> pageQueryMaps(QueryWrapper where) {
        return execMapPageQuery(where);
    }

    public Page<Map> pageSelectQueryMap(Pageable pageable, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(select);
        wrapper.setPageable(pageable);
        return execMapPageQuery(wrapper);
    }

    public Page pageQueryObjects(Pageable pageable, QueryWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(where.getSelectSql());
        wrapper.setPageable(pageable);
        return execObjectPageQuery(wrapper);
    }
    public Page pageQueryObjects(QueryWrapper where) {
        return execObjectPageQuery(where);
    }
    public Page pageQueryObjects(Pageable pageable, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setSelectSql(select);
        wrapper.setPageable(pageable);
        return execObjectPageQuery(wrapper);
    }

    public <C>  Page<C> pageQueryColumns(Class<C> cClass, Pageable pageable, QueryWrapper where) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where);
        wrapper.setSelectSql(where.getSelectSql());
        wrapper.setPageable(pageable);
        return execColumnPageQuery(cClass,wrapper);
    }
    public <C>  Page<C> pageQueryColumns(Class<C> cClass, QueryWrapper where) {
        return execColumnPageQuery(cClass,where);
    }
    public <C>  Page<C> pageQueryColumns(Class<C> cClass, Pageable pageable, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setSelectSql(select);
        wrapper.setPageable(pageable);
        return execColumnPageQuery(cClass,wrapper);
    }
    //</editor-fold>
    //</editor-fold>
    //</editor-fold>

    //<editor-fold desc="wrapper update">
    @Override
    public int update(UpdateWrapper wrapper) {
        return execUpdate(wrapper);
    }
    //</editor-fold>
    //<editor-fold desc="wrapper delete">
    @Override
    public int delete(WhereWrapper wrapper) {
        DeleteWrapper deleteWrapper=new DeleteWrapper(getTableName(), wrapper);
        return execDelete(deleteWrapper);
    }
    //</editor-fold>
    //<editor-fold desc="wrapper query">
    @Override
    public T getModel(WhereWrapper wrapper) {
        QueryWrapper queryWrapper=new QueryWrapper(getTableName(),wrapper);
        return execModelQuery(queryWrapper);
    }

    @Override
    public List<T> findModels(WhereWrapper wrapper) {
        QueryWrapper queryWrapper=new QueryWrapper(getTableName(),wrapper);
        return execListModelQuery(queryWrapper);
    }

    @Override
    public Page<T> pageModels(WhereWrapper wrapper) {
        QueryWrapper queryWrapper=new QueryWrapper(getTableName(),wrapper);
        return execPageQuery(queryWrapper);
    }

    @Override
    public <C> C getColumn(Class<C> cClass, QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())){
            wrapper.setTableName(getTableName());
        }
        return execColumnQuery(cClass,wrapper);
    }

    @Override
    public <C> List<C> findColumns(Class<C> cClass, QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())){
            wrapper.setTableName(getTableName());
        }
        return execListColumnQuery(cClass,wrapper);
    }

    @Override
    public <C> Page<C> pageColumns(Class<C> cClass, QueryWrapper wrapper) {
        return execColumnPageQuery(cClass,wrapper);
    }
    @Override
    public Map getMap(QueryWrapper wrapper) {
        return execMapQuery(wrapper);
    }
    @Override
    public List<Map> findMaps(QueryWrapper wrapper) {
        return execListMapQuery(wrapper);
    }
    @Override
    public Page<Map> pageMaps(QueryWrapper wrapper) {
        return execMapPageQuery(wrapper);
    }



    //</editor-fold>

    //<editor-fold desc="execPage">
    @Override
    public Page execObjectPageQuery(QueryWrapper wrapper) {
        return execObjectPageQuery( wrapper.getPageNumber(),wrapper.getPageSize(),wrapper.getSelectSql(),wrapper.getWhereSql(),wrapper.getSqlParams());
    }
    @Override
    public Page execObjectPageQuery(int pageNumber, int pageSize, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setPageNumber(pageNumber);
        wrapper.setPageSize(pageSize);
        wrapper.setSelectSql(select);

        Long totalRecord = execColumnQuery(Long.class, wrapper.getCountSql(), wrapper.getSqlParams());
        List list = execListObjectQuery( wrapper.getPageSql(), wrapper.getSqlParams());
        Pageable page=PageRequest.of(pageNumber,pageSize);
        return new PageImpl<>(list, page, totalRecord);
    }
    @Override
    public Page<T> execPageQuery(int pageNumber,int pageSize,String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setPageNumber(pageNumber);
        wrapper.setPageSize(pageSize);

        Long totalRecord = execColumnQuery(Long.class, wrapper.getCountSql(), wrapper.getSqlParams());
        List<T> list = execListModelQuery(wrapper.getPageSql(), wrapper.getSqlParams());
        Pageable page=PageRequest.of(pageNumber,pageSize);
        return new PageImpl<>(list, page, totalRecord);
    }

    @Override
    public <C> Page<C> execColumnPageQuery(int pageNumber, int pageSize, Class<C> tClass, String select, String where, Object... params) {
        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setPageNumber(pageNumber);
        wrapper.setPageSize(pageSize);
        wrapper.setSelectSql(select);

        Long totalRecord = execColumnQuery(Long.class, wrapper.getCountSql(), wrapper.getSqlParams());
        List<C> list = execListColumnQuery(tClass, wrapper.getPageSql(), wrapper.getSqlParams());
        Pageable page=PageRequest.of(pageNumber,pageSize);
        return new PageImpl<>(list, page, totalRecord);
    }

    @Override
    public Page<Map> execMapPageQuery(int pageNumber,int pageSize,String select,String where, Object... params) {

        QueryWrapper wrapper=new QueryWrapper(getTableName(),where,params);
        wrapper.setSelectSql(select);
        wrapper.setPageNumber(pageNumber);
        wrapper.setPageSize(pageSize);
        wrapper.setSelectSql(select);

        Long totalRecord = execColumnQuery(Long.class, wrapper.getCountSql(), wrapper.getSqlParams());
        List<Map> list = execListMapQuery( wrapper.getPageSql(), wrapper.getSqlParams());
        Pageable page=PageRequest.of(pageNumber,pageSize);
        return new PageImpl<>(list, page, totalRecord);
    }

    @Override
    public Page<T> execPageQuery(QueryWrapper wrapper) {
        return execPageQuery(wrapper.getPageNumber(),wrapper.getPageSize(),wrapper.getSelectSql(),wrapper.getWhereSql(),wrapper.getSqlParams());
    }

    @Override
    public <C> Page<C> execColumnPageQuery(Class<C> tClass, QueryWrapper wrapper) {
        return execColumnPageQuery( wrapper.getPageNumber(),wrapper.getPageSize(),tClass,wrapper.getSelectSql(),wrapper.getWhereSql(),wrapper.getSqlParams());
    }

    @Override
    public Page<Map> execMapPageQuery(QueryWrapper wrapper) {
        return execMapPageQuery(wrapper.getPageNumber(),wrapper.getPageSize(),wrapper.getSelectSql(),wrapper.getWhereSql(),wrapper.getSqlParams());
    }

    //</editor-fold>
    //<editor-fold desc="exec wrapper">
    @Override
    public int execUpdate(UpdateWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execUpdate(wrapper.getSql(),wrapper.getSqlParams());
    }
    @Override
    public int execDelete(DeleteWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execUpdate(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public Object execObjectQuery( QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execObjectQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public List execListObjectQuery(QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execListObjectQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public <C> C execColumnQuery(Class<C> tClass, QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execColumnQuery(tClass,wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public <C> List<C> execListColumnQuery(Class<C> tClass, QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execListColumnQuery(tClass,wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public Map execMapQuery(QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execMapQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public List<Map> execListMapQuery(QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execListMapQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public T execModelQuery(QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execModelQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    @Override
    public List<T> execListModelQuery(QueryWrapper wrapper) {
        if(StringUtils.isBlank(wrapper.getTableName())) wrapper.setTableName(getTableName());
        return execListModelQuery(wrapper.getSql(),wrapper.getSqlParams());
    }

    //</editor-fold>
    //<editor-fold desc="exec">
    @Override
    public int execUpdate(String sql, Object... params) {
        Query query = entityManager.createNativeQuery(sql);
        if (params != null) {
            int index = 1;
            for (Object param : params) {
                query.setParameter(index, param);
                index++;
            }
        }
        return query.executeUpdate();
    }

    @Override
    public Object execObjectQuery( String sql, Object... params) {
        List list = execListObjectQuery( sql, params);
        if(list.size()==0){
            return null;
        }
        return list.get(0);
    }

    @Override
    public List execListObjectQuery( String sql, Object... params) {
        Query query = entityManager.createNativeQuery(sql);
        if (params != null) {
            int index = 1;
            for (Object param : params) {
                query.setParameter(index, param);
                index++;
            }
        }
//        NativeQueryImplementor nativeQueryImplementor = query.unwrap(NativeQueryImpl.class);
//                .setResultTransformer(new ColumnToBean(resultClass));
        return query.getResultList();
    }

    @Override
    public <C> C execColumnQuery(Class<C> tClass, String sql, Object... params) {
        List<C> list = execListColumnQuery(tClass, sql, params);
        if(list.size()==0){
            return null;
        }
        return list.get(0);
    }

    @Override
    public <C> List<C> execListColumnQuery(Class<C> tClass, String sql, Object... params) {
        Query query = entityManager.createNativeQuery(sql,tClass);
        if (params != null) {
            int index = 1;
            for (Object param : params) {
                query.setParameter(index, param);
                index++;
            }
        }
//        NativeQueryImplementor nativeQueryImplementor = query.unwrap(NativeQueryImpl.class);
//                .setResultTransformer(new ColumnToBean(resultClass));
        return query.getResultList();
    }

    @Override
    public Map execMapQuery(String sql, Object... params) {
        List<Map> list = execListMapQuery( sql, params);
        if(list.size()==0){
            return null;
        }
        return list.get(0);
    }

    @Override
    public List<Map> execListMapQuery(String sql, Object... params) {
        Query query = entityManager.createNativeQuery(sql);
        if (params != null) {
            int index = 1;
            for (Object param : params) {
                query.setParameter(index, param);
                index++;
            }
        }
        List<Map> result = query
                .unwrap(SQLQuery.class)
                .setResultTransformer(Transformers.ALIAS_TO_ENTITY_MAP)
                .list();
        return result;
    }

    @Override
    public T execModelQuery(String sql, Object... params) {
        List<T> list = execListModelQuery( sql, params);
        if(list.size()==0){
            return null;
        }
        return list.get(0);
    }

    @Override
    public List<T> execListModelQuery(String sql, Object... params) {
        Query query = entityManager.createNativeQuery(sql,entityClass);
        if (params != null) {
            int index = 1;
            for (Object param : params) {
                query.setParameter(index, param);
                index++;
            }
        }
//        NativeQueryImplementor nativeQueryImplementor = query.unwrap(NativeQueryImpl.class);
//                .setResultTransformer(new ColumnToBean(resultClass));
        return query.getResultList();
    }

    //</editor-fold>
    @Override
    public void batchInvokeEvent(Collection targets, Event event) {

    }

    @Override
    public void invokeEvent(Object target, Event event) {

    }
    protected String getTableName() {
        SessionFactoryImpl  sessionFactory = (SessionFactoryImpl)entityManager.getEntityManagerFactory().unwrap(SessionFactory.class);
        SingleTableEntityPersister entityPersister = (SingleTableEntityPersister )sessionFactory.getEntityPersister(entityClass.getName());
        if(entityPersister!=null){
            return entityPersister.getTableName();
        }
        String name = "";
        Table tableAnnotation = (Table) entityClass.getAnnotation(Table.class);
        if (tableAnnotation != null) {
            name = tableAnnotation.name();
        }
        if (name != null && !name.isEmpty()) {
            return name;
        }
        Entity entityAnnotation = (Entity) entityClass.getAnnotation(Entity.class);
        if (entityAnnotation != null) {
            name = entityAnnotation.name();
        }
        if (name != null && !name.isEmpty()) {
            return name;
        }
        String simpleName = entityClass.getSimpleName();
        return simpleName;
    }
    protected String getTableName(Class entityClass) {
        SessionFactoryImpl  sessionFactory = (SessionFactoryImpl)entityManager.getEntityManagerFactory().unwrap(SessionFactory.class);
        SingleTableEntityPersister entityPersister = (SingleTableEntityPersister )sessionFactory.getEntityPersister(entityClass.getName());
        if(entityPersister!=null){
            return entityPersister.getTableName();
        }
        String name = "";
        Table tableAnnotation = (Table) entityClass.getAnnotation(Table.class);
        if (tableAnnotation != null) {
            name = tableAnnotation.name();
        }
        if (name != null && !name.isEmpty()) {
            return name;
        }
        Entity entityAnnotation = (Entity) entityClass.getAnnotation(Entity.class);
        if (entityAnnotation != null) {
            name = entityAnnotation.name();
        }
        if (name != null && !name.isEmpty()) {
            return name;
        }
        String simpleName = entityClass.getSimpleName();
        return simpleName;
    }

}
