package cn.navigational;

import cn.navigational.annotation.Application;
import cn.navigational.annotation.ScanPackage;
import cn.navigational.annotation.Verticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Vertx;
import io.vertx.core.file.FileSystem;
import io.vertx.core.json.JsonObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;


public class VertxApplication {

    //config file path
    private String config;

    //vertx instance
    private final Vertx vertx = Vertx.vertx();

    //logger
    private final Logger logger = LogManager.getLogger();

    //vertx deployment options
    private final DeploymentOptions options = new DeploymentOptions();

    //Installed in the scanned class path
    private final List<String> verticle = new ArrayList<>();

    //Packet paths to be scanned
    private String[] packages;

    //json config
    private JsonObject appConfig = new JsonObject();


    //init application
    public void init() {
        final FileSystem fs = vertx.fileSystem();

        final Application app = this.getClass().getDeclaredAnnotation(Application.class);
        final ScanPackage scanPackage = this.getClass().getDeclaredAnnotation(ScanPackage.class);

        if (app != null) {
            config = app.config();
        }

        if (scanPackage != null) {
            packages = scanPackage.packages();
        }

        if (config != null) {
            appConfig = fs.readFileBlocking(config).toJsonObject();
        }

        if (packages == null || packages.length == 0) {
            logger.warn("No packages need to be scanned");
            packages = new String[0];
        }

        //Start scanning packet paths
        scanPackage(packages);

        options.setConfig(appConfig);
        if (!verticle.isEmpty()) {
            verticle.forEach(_clazz -> {
                logger.info("start deploy {}", _clazz);
                vertx.deployVerticle(_clazz, options, _rs -> {
                    if (_rs.succeeded()) {
                        logger.info("{} deployment success!", _clazz);
                    } else {
                        _rs.cause().printStackTrace();
                        logger.error("{} deployment failed:{}", _clazz, _rs.cause().getMessage());
                    }
                });
            });
        }
    }

    //Scanning class annotated with @Verticle annotation
    private void scanPackage(String... pack) {
        if (pack.length == 0) {
            return;
        }
        for (String path : pack) {
            try {
                final Enumeration<URL> clazz = getClass().getClassLoader().getResources(path.replaceAll("\\.", "/"));
                while (clazz.hasMoreElements()) {
                    final URL url = clazz.nextElement();
                    if (url != null) {
                        //FileSystem
                        if (url.getProtocol().equals("file")) {
                            logger.info("scan type {}", "FileSystem");
                            final String packagePath = URLDecoder.decode(url.getFile(), String.valueOf(StandardCharsets.UTF_8));
                            addClass(packagePath, path);
                        } else {
                            //Jar
                            logger.info("scan type:{}", "jar");
                            verticle.addAll(scanClassFromJar(url, path));
                        }
                    }
                }
            } catch (IOException e) {
                e.printStackTrace();
                logger.error("scan package failed:{}", e.getCause().getMessage());
            }
        }
    }

    //Read the class path from jar
    private List<String> scanClassFromJar(final URL url, final String path) throws IOException {
        final List<String> list = new ArrayList<>();
        final JarURLConnection jarURLConnection = (JarURLConnection) url.openConnection();
        if (Objects.nonNull(jarURLConnection) && Objects.nonNull(jarURLConnection.getJarFile())) {
            final JarFile jarFile = jarURLConnection.getJarFile();
            final Enumeration<JarEntry> entries = jarFile.entries();
            while (entries.hasMoreElements()) {
                final JarEntry jarEntry = entries.nextElement();
                final String jarEntryName = jarEntry.getName();
                final String prefix = jarEntryName.replaceAll("/", "\\.");
                if (prefix.startsWith(path) && jarEntryName.endsWith(".class")) {
                    final String className = jarEntryName.substring(0, jarEntryName.lastIndexOf(".")).replaceAll("/", "\\.");
                    final boolean cl = isFitClass(className);
                    if (cl) {
                        logger.info(className);
                        list.add(className);
                    }
                }
            }

        }
        return list;
    }

    //Add class to list
    private void addClass(String packPath, String packName) {
        final File[] files = new File(packPath).listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return dir.isFile() && name.endsWith(".class") || dir.isDirectory();
            }
        });
        if (files == null) {
            return;
        }
        for (File file : files) {
            final String fileName = file.getName();
            if (file.isFile()) {
                final String className = packName + "." + fileName.split("\\.")[0];
                final boolean clazz = isFitClass(className);
                if (clazz) {
                    verticle.add(className);
                }
            } else {
                final String subPath = packPath + File.separator + fileName;
                final String subPackage = packName + "." + fileName;
                addClass(subPath, subPackage);
            }
        }
    }

    //Determine whether to annotate the @Verticle annotation
    private boolean isFitClass(String className) {
        final Class clazz;
        try {
            clazz = Class.forName(className);
            if (clazz.getDeclaredAnnotation(Verticle.class) != null) {
                return true;
            }
        } catch (ClassNotFoundException e) {
            logger.error("class {} not found!", className);
        }
        return false;
    }

    //Setting configuration file path
    public VertxApplication setConfig(String config) {
        this.config = config;
        return this;
    }


    public DeploymentOptions getOptions() {
        return options;
    }

    public Vertx getVertx() {
        return vertx;
    }

    public String[] getPackages() {
        return packages;
    }

    public void setPackages(String... packages) {
        this.packages = packages;
    }

    public JsonObject getAppConfig() {
        return appConfig;
    }

    public void setAppConfig(JsonObject appConfig) {
        this.appConfig = appConfig;
    }
}
