package cn.morethank.open.admin.system.service.impl;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.service.RedisService;
import cn.morethank.open.admin.common.service.RsaService;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.*;
import cn.morethank.open.admin.system.mapper.*;
import cn.morethank.open.admin.system.service.SysMenuService;
import cn.morethank.open.admin.system.service.SysRoleService;
import cn.morethank.open.admin.system.service.SysUserService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.AllArgsConstructor;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户信息表 服务实现类
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Service
public class SysUserServiceImpl extends ServiceImpl<SysUserMapper, SysUser> implements SysUserService {

    private final SysUserMapper sysUserMapper;
    private final SysRoleMapper sysRoleMapper;
    private final SysPostMapper sysPostMapper;

    private final RedisService redisService;

    private final SysRoleService sysRoleService;
    private final SysMenuService sysMenuService;
	private final SysUserPostMapper sysUserPostMapper;
	private final SysUserRoleMapper sysUserRoleMapper;
    private final RsaService rsaService;

    @Override
    public SysUser getByUsername(String username) {
        return sysUserMapper.selectUserByUserName(username);
    }

	@Override
	public IPage<SysUser> selectListPage(Page<SysUser> page, SysUser sysUser) {
		return sysUserMapper.selectListPage(page, sysUser);
	}

    @Override
    public String getUserAuthorityInfo(Long userId) {
		SysUser sysUser = sysUserMapper.selectById(userId);

		//  ROLE_admin,ROLE_normal,sys:user:list,....
		String authority = "";

		if (redisService.hasKey("GrantedAuthority:" + sysUser.getUserName())) {
			authority = (String) redisService.get("GrantedAuthority:" + sysUser.getUserName());

		} else {
			// 获取角色编码
			List<SysRole> roles = sysRoleService.list(new QueryWrapper<SysRole>()
					.inSql("role_id", "select role_id from sys_user_role where user_id = " + userId));

			if (roles.size() > 0) {
				String roleCodes = roles.stream().map(r -> "ROLE_" + r.getRoleKey()).collect(Collectors.joining(","));
				authority = roleCodes.concat(",");
			}

			// 获取菜单操作编码
			List<Long> menuIds = sysUserMapper.getNavMenuIds(userId);
			if (menuIds.size() > 0) {
				List<SysMenu> menus = sysMenuService.listByIds(menuIds);
				String menuPerms = menus.stream().map(m -> m.getPerms()).collect(Collectors.joining(","));

				authority = authority.concat(menuPerms);
			}

			redisService.set("GrantedAuthority:" + sysUser.getUserName(), authority, 60 * 60);
		}

		return authority;
    }

	@Override
	public SysUser selectUserById(Long userId) {
		return sysUserMapper.selectUserById(userId);
	}

	@Override
	public void checkUserAllowed(SysUser sysUser) {
        if (StringUtils.isNotNull(sysUser.getUserId()) && sysUser.isAdmin()) {
            throw new ServiceException("不允许操作超级管理员用户");
        }
	}

	@Override
	public int deleteUserByIds(Long[] userIds) {
        for (Long userId : userIds) {
            checkUserAllowed(new SysUser(userId));
        }
        // 删除用户与角色关联
        sysUserRoleMapper.deleteUserRole(userIds);
        // 删除用户与岗位关联
        sysUserPostMapper.deleteUserPost(userIds);

		LambdaUpdateWrapper<SysUser> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
		ArrayList<Long> arrayList = new ArrayList<>(userIds.length);
		Collections.addAll(arrayList, userIds);
        lambdaUpdateWrapper.in(SysUser::getUserId, arrayList).set(SysUser::getDelFlag, GlobalConstant.DELETED);
        return sysUserMapper.update(null, lambdaUpdateWrapper);
	}

	@Override
	public boolean checkUserNameUnique(String userName) {
        LambdaQueryWrapper<SysUser> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysUser::getUserName, userName);
        long count = sysUserMapper.selectCount(wrapper);
        return count > 0;
	}

	@Override
	public boolean checkPhoneUnique(SysUser sysUser) {
        LambdaQueryWrapper<SysUser> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysUser::getPhoneNumber, sysUser.getPhoneNumber());
        if(sysUser.getUserId() != null && sysUser.getUserId() > 0L) {
            wrapper.ne(SysUser::getUserId, sysUser.getUserId());
        }
        long count = sysUserMapper.selectCount(wrapper);
        return count > 0;
	}

	@Override
	public boolean checkEmailUnique(SysUser sysUser) {
        LambdaQueryWrapper<SysUser> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysUser::getEmail, sysUser.getEmail());
        if(sysUser.getUserId() != null && sysUser.getUserId() > 0L) {
            wrapper.ne(SysUser::getUserId, sysUser.getUserId());
        }
        long count = sysUserMapper.selectCount(wrapper);
        return count > 0;
	}

	@Override
	@Transactional(rollbackFor = Exception.class)
	public int insertUser(SysUser sysUser) {
        // 新增用户信息
        int rows = sysUserMapper.insert(sysUser);
        // 新增用户岗位关联
        insertUserPost(sysUser);
        // 新增用户与角色管理
        insertUserRole(sysUser);
        return rows;
	}

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int updateUser(SysUser sysUser) {
        Long userId = sysUser.getUserId();
        // 删除用户与角色关联
        sysUserRoleMapper.deleteUserRoleByUserId(userId);
        // 新增用户与角色管理
        insertUserRole(sysUser);
        // 删除用户与岗位关联
        sysUserPostMapper.deleteUserPostByUserId(userId);
        // 新增用户与岗位管理
        insertUserPost(sysUser);
        return sysUserMapper.updateById(sysUser);
    }

    @Override
    public int resetPwd(SysUser sysUser) {
        // 根据uuid从缓存中获取私钥
        String privateKey = (String) redisService.hget(GlobalConstant.PRIVATE_KEY, sysUser.getUuid());
        // 解密password得到明文
        String rawPasswd = rsaService.privateDecrypt(sysUser.getPassword(), privateKey);
        log.warn(rawPasswd);
        String passwd = new BCryptPasswordEncoder().encode(rawPasswd);
        LambdaUpdateWrapper<SysUser> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
        lambdaUpdateWrapper.eq(SysUser::getUserId, sysUser.getUserId()).set(SysUser::getPassword, passwd);
        return sysUserMapper.update(null, lambdaUpdateWrapper);
    }

    @Override
    public String selectUserRoleGroup(String userName) {
        List<SysRole> list = sysRoleMapper.selectRolesByUserName(userName);
        if (CollectionUtils.isEmpty(list)) {
            return StringUtils.EMPTY;
        }
        return list.stream().map(SysRole::getRoleName).collect(Collectors.joining(","));
    }

    @Override
    public String selectUserPostGroup(String userName) {
        List<SysPost> list = sysPostMapper.selectPostsByUserName(userName);
        if (CollectionUtils.isEmpty(list)) {
            return StringUtils.EMPTY;
        }
        return list.stream().map(SysPost::getPostName).collect(Collectors.joining(","));
    }

    @Override
    public void insertUserAuth(Long userId, Long[] roleIds) {
        sysUserRoleMapper.deleteUserRoleByUserId(userId);
        insertUserRole(userId, roleIds);
    }

    @Override
    public IPage<SysUser> selectAllocatedList(Page<SysUser> page, SysUser sysUser) {
        return sysUserMapper.selectAllocatedList(page, sysUser);
    }

    @Override
    public IPage<SysUser> selectUnallocatedList(Page<SysUser> page, SysUser sysUser) {
        return sysUserMapper.selectUnallocatedList(page, sysUser);
    }

    @Override
    public List<SysUser> selectUserList(SysUser sysUser) {
        return sysUserMapper.selectUserList(sysUser);
    }

    @Override
    public boolean updateUserAvatar(String userName, String avatar) {
        return sysUserMapper.updateUserAvatar(userName, avatar) > 0;
    }

    public void insertUserRole(SysUser sysUser) {
        this.insertUserRole(sysUser.getUserId(), sysUser.getRoleIds());
    }

    /**
     * 新增用户角色信息
     *
     * @param userId 用户ID
     * @param roleIds 角色组
     */
    public void insertUserRole(Long userId, Long[] roleIds) {
        if (StringUtils.isNotEmpty(roleIds)) {
            // 新增用户与角色管理
            List<SysUserRole> list = new ArrayList<>(roleIds.length);
            for (Long roleId : roleIds) {
                SysUserRole ur = new SysUserRole();
                ur.setUserId(userId);
                if(roleId == null) {
                    continue;
                }
                ur.setRoleId(roleId);
                list.add(ur);
            }
            sysUserRoleMapper.batchUserRole(list);
        }
    }

    private void insertUserPost(SysUser sysUser) {
        Long[] posts = sysUser.getPostIds();
        if (StringUtils.isNotEmpty(posts)) {
            // 新增用户与岗位管理
            List<SysUserPost> list = new ArrayList<>(posts.length);
            for (Long postId : posts) {
                SysUserPost up = new SysUserPost();
                up.setUserId(sysUser.getUserId());
                up.setPostId(postId);
                list.add(up);
            }
            sysUserPostMapper.batchUserPost(list);
        }
    }

}
