package cn.morethank.open.admin.system.service.impl;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysRole;
import cn.morethank.open.admin.system.domain.SysRoleMenu;
import cn.morethank.open.admin.system.domain.SysUserRole;
import cn.morethank.open.admin.system.mapper.SysRoleMapper;
import cn.morethank.open.admin.system.mapper.SysRoleMenuMapper;
import cn.morethank.open.admin.system.mapper.SysUserRoleMapper;
import cn.morethank.open.admin.system.service.SysRoleService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.*;

/**
 * 角色信息表 服务实现类
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Service
public class SysRoleServiceImpl extends ServiceImpl<SysRoleMapper, SysRole> implements SysRoleService {

    private final SysRoleMapper sysRoleMapper;
    private final SysRoleMenuMapper sysRoleMenuMapper;
    private final SysUserRoleMapper sysUserRoleMapper;
    private final JwtService jwtService;

    @Override
    public IPage<SysRole> selectListPage(Page<SysRole> page, LambdaQueryWrapper<SysRole> query) {
        return sysRoleMapper.selectPage(page, query);
    }

    @Override
    public Collection<String> selectRolePermissionByUserId(Long userId) {
        List<SysRole> perms = sysRoleMapper.selectRolePermissionByUserId(userId);
        Set<String> permsSet = new HashSet<>();
        for (SysRole perm : perms) {
            if (perm != null) {
                permsSet.addAll(Arrays.asList(perm.getRoleKey().trim().split(GlobalConstant.COMMA)));
            }
        }
        return permsSet;
    }

    @Override
    public int updateRoleStatus(SysRole role) {
        LambdaUpdateWrapper<SysRole> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
        lambdaUpdateWrapper.eq(SysRole::getRoleId, role.getRoleId()).set(SysRole::getStatus, role.getStatus())
                .set(SysRole::getUpdateBy, jwtService.getUserName()).set(SysRole::getUpdateTime, LocalDateTime.now());
        return sysRoleMapper.update(null, lambdaUpdateWrapper);
    }

    @Override
    public void checkRoleAllowed(SysRole role) {
        if (StringUtils.isNotNull(role.getRoleId()) && role.isAdmin()) {
            throw new ServiceException("不允许对超级管理员角色做操作");
        }
    }

    @Override
    public boolean checkRoleNameUnique(SysRole role) {
        LambdaQueryWrapper<SysRole> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysRole::getDelFlag, GlobalConstant.UNDELETED).eq(SysRole::getRoleName, role.getRoleName());
        if(role.getRoleId() != null && role.getRoleId() > 0L) {
            wrapper.ne(SysRole::getRoleId, role.getRoleId());
        }
        long count = sysRoleMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public boolean checkRoleKeyUnique(SysRole role) {
        LambdaQueryWrapper<SysRole> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysRole::getDelFlag, GlobalConstant.UNDELETED).eq(SysRole::getRoleKey, role.getRoleKey());
        if(role.getRoleId() != null && role.getRoleId() > 0L) {
            wrapper.ne(SysRole::getRoleId, role.getRoleId());
        }
        long count = sysRoleMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public int insertRole(SysRole role) {
        // 新增角色信息
        sysRoleMapper.insert(role);
        return insertRoleMenu(role);
    }

    @Override
    public int deleteRoleByIds(Long[] roleIds) {
        for (Long roleId : roleIds) {
            checkRoleAllowed(new SysRole(roleId));
            SysRole role = getById(roleId);
            if (countUserRoleByRoleId(roleId) > 0) {
                throw new ServiceException(String.format("%1$s已分配,不能删除", role.getRoleName()));
            }
        }
        // 删除角色与菜单关联
        sysRoleMenuMapper.deleteRoleMenu(roleIds);
        // 删除角色与部门关联
        sysRoleMenuMapper.deleteRoleDept(roleIds);
        ArrayList<Long> arrayList = new ArrayList<>(roleIds.length);
		Collections.addAll(arrayList, roleIds);
        return sysRoleMapper.deleteBatchIds(arrayList);
    }

    /**
     * 通过角色ID查询角色使用数量
     *
     * @param roleId 角色ID
     * @return 结果
     */
    @Override
    public int countUserRoleByRoleId(Long roleId) {
        return sysRoleMenuMapper.countUserRoleByRoleId(roleId);
    }

    @Override
    public int updateRole(SysRole role) {
        // 修改角色信息
        sysRoleMapper.updateRole(role);
        // 删除角色与菜单关联
        sysRoleMenuMapper.deleteRoleMenuByRoleId(role.getRoleId());
        return insertRoleMenu(role);
    }

    @Override
    public List<SysRole> selectRolesByUserId(Long userId) {
        List<SysRole> userRoles = sysRoleMapper.selectRolePermissionByUserId(userId);
        List<SysRole> roles = selectRoleList(new SysRole());
        for (SysRole role : roles) {
            for (SysRole userRole : userRoles) {
                if (role.getRoleId().longValue() == userRole.getRoleId().longValue()) {
                    role.setFlag(true);
                    break;
                }
            }
        }
        return roles;
    }

    @Override
    public List<SysRole> selectRoleList(SysRole sysRole) {
        LambdaQueryWrapper<SysRole> query = getQueryWrapper(sysRole);
        return sysRoleMapper.selectList(query);
    }

    @Override
    public int insertAuthUsers(Long roleId, Long[] userIds) {
        // 新增用户与角色管理
        List<SysUserRole> list = new ArrayList<>();
        for (Long userId : userIds) {
            SysUserRole ur = new SysUserRole();
            ur.setUserId(userId);
            ur.setRoleId(roleId);
            list.add(ur);
        }
        return sysUserRoleMapper.batchUserRole(list);
    }

    @Override
    public int deleteAuthUser(SysUserRole userRole) {
        return sysUserRoleMapper.deleteUserRoleInfo(userRole);
    }

    @Override
    public int deleteAuthUsers(Long roleId, Long[] userIds) {
        return sysUserRoleMapper.deleteUserRoleInfos(roleId, userIds);
    }

    /**
     * 新增角色菜单信息
     *
     * @param role 角色对象
     */
    public int insertRoleMenu(SysRole role) {
        int rows = 1;
        // 新增用户与角色管理
        List<SysRoleMenu> list = new ArrayList<SysRoleMenu>();
        for (Long menuId : role.getMenuIds()) {
            SysRoleMenu rm = new SysRoleMenu();
            rm.setRoleId(role.getRoleId());
            rm.setMenuId(menuId);
            list.add(rm);
        }

        if (list.size() > 0) {
            rows = sysRoleMenuMapper.batchRoleMenu(list);
        }
        return rows;
    }

    private LambdaQueryWrapper<SysRole> getQueryWrapper(SysRole sysRole) {
        LambdaQueryWrapper<SysRole> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysRole.getRoleName())) {
            query.like(SysRole::getRoleName, sysRole.getRoleName());
        }
        if (StrUtil.isNotEmpty(sysRole.getRoleKey())) {
            query.like(SysRole::getRoleKey, sysRole.getRoleKey());
        }
        if (StrUtil.isNotEmpty(sysRole.getStatus())) {
            query.eq(SysRole::getStatus, sysRole.getStatus());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysRole.getParams());
        // 排序
        query.orderByAsc(SysRole::getRoleSort);
        return query;
    }
}
