package cn.morethank.open.admin.system.service.impl;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysGenColumn;
import cn.morethank.open.admin.system.domain.SysGenTable;
import cn.morethank.open.admin.system.mapper.SysGenColumnMapper;
import cn.morethank.open.admin.system.mapper.SysGenTableMapper;
import cn.morethank.open.admin.common.service.FreemarkerService;
import cn.morethank.open.admin.system.service.SysGenColumnService;
import cn.morethank.open.admin.system.service.SysGenTableService;
import cn.morethank.open.admin.common.util.GenConstants;
import cn.morethank.open.admin.common.util.GenUtils;
import com.alibaba.fastjson2.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import freemarker.template.Configuration;
import freemarker.template.Template;
import lombok.AllArgsConstructor;
import org.apache.commons.io.IOUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.ByteArrayOutputStream;
import java.io.StringWriter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 代码生成业务表 服务实现类
 *
 * @author morethank
 * @since 2023-01-08 00:30:18
 */
@AllArgsConstructor
@Service
public class SysGenTableServiceImpl extends ServiceImpl<SysGenTableMapper, SysGenTable> implements SysGenTableService {

    private final SysGenTableMapper sysGenTableMapper;
    private final SysGenColumnMapper sysGenColumnMapper;
    private final SysGenColumnService sysGenColumnService;
    private final Configuration freemarkerConfiguration;
    private final FreemarkerService freemarkerService;

    @Override
    public IPage<SysGenTable> selectListPage(Page<SysGenTable> page, LambdaQueryWrapper<SysGenTable> query) {
        return sysGenTableMapper.selectPage(page, query);
    }

    @Override
    public int deleteByIds(Long[] ids) {
        ArrayList<Long> arrayList = new ArrayList<>(ids.length);
		Collections.addAll(arrayList, ids);
        this.removeByIds(arrayList);
        return ids.length;
    }

    @Override
    public void genTableList() {
        List<Map<String, String>> tables = sysGenTableMapper.getTableList();
        List<SysGenTable> sysGenTables = new ArrayList<>(tables.size());
        for (Map<String, String> tableMap: tables) {
            SysGenTable table = new SysGenTable();
            String comment = tableMap.get("table_comment");
            table.setTableComment(comment);
            String tableName = tableMap.get("table_name");
            table.setTableName(tableName);
            table.setClassName(StringUtils.underlineToCamel(tableName));
            table.setPackageName("cn.morethank.open.admin");
            table.setModuleName("system");
            table.setBusinessName(tableName);
            table.setFunctionName(comment.endsWith("表") ? comment.substring(0, comment.length() - 1) : comment);
            sysGenTables.add(table);
        }
        sysGenTableMapper.batchInsertTable(sysGenTables);
    }

    @Override
    public Map<String, String> previewCode(Long tableId) {
        Map<String, String> dataMap = new LinkedHashMap<>();
        // 查询表信息
        SysGenTable table = sysGenTableMapper.selectGenTableById(tableId);
        List<SysGenColumn> columns = table.getColumns();
        if(StringUtils.isEmpty(columns) || columns.size() <= 1) {
            sysGenColumnService.genColumnListByTableName(tableId, table.getTableName());
            table = sysGenTableMapper.selectGenTableById(tableId);
        }
        // 设置主子表信息
        setSubTable(table);
        // 设置主键列信息
        setPkColumn(table);

        Map<String, Object> objectMap = freemarkerService.getObjectMap(table);

        // 获取模板列表
        List<String> templates = freemarkerService.getTemplateList(table.getTplCategory());
        for (String template : templates) {
            // 渲染模板
            StringWriter sw = new StringWriter();
            try {
                Template tpl = freemarkerConfiguration.getTemplate(template, GlobalConstant.UTF8);
                tpl.process(objectMap, sw);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            dataMap.put(template, sw.toString());
        }
        return dataMap;
    }

    @Override
    public List<SysGenTable> selectGenTableAll() {
        return sysGenTableMapper.selectGenTableAll();
    }

    @Override
    public void updateGenTable(SysGenTable genTable) {
        String options = JSON.toJSONString(genTable.getParams());
        genTable.setOptions(options);
        boolean update = updateById(genTable);
        if (update) {
            for (SysGenColumn cenTableColumn : genTable.getColumns()) {
                sysGenColumnMapper.updateById(cenTableColumn);
            }
        }
    }

    @Override
    public byte[] downloadCode(String[] tableNames) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream zip = new ZipOutputStream(outputStream);
        for (String tableName : tableNames) {
            generatorCode(tableName, zip);
        }
        IOUtils.closeQuietly(zip);
        return outputStream.toByteArray();
    }

    @Override
    public void synchDb(String tableName) {
        // 查表结构中的字段, 如果表不存在则报错
        List<Map<String, Object>> tableColumns = sysGenColumnMapper.getTableColumns(tableName);
        if(CollectionUtils.isEmpty(tableColumns)) {
            throw new ServiceException("同步失败，表结构不存在");
        }

        SysGenTable sysGenTable = sysGenTableMapper.selectGenTableByName(tableName);
        List<SysGenColumn> oldColumns = sysGenTable.getColumns();
        Map<String, SysGenColumn> tableColumnMap = oldColumns.stream().collect(Collectors.toMap(SysGenColumn::getColumnName, Function.identity()));
        for (Map<String, Object> tableMap: tableColumns) {
            String columnName = (String) tableMap.get("column_name");
            SysGenColumn column = tableColumnMap.get(columnName);
            if(column == null) {
                // 新增字段
                column = new SysGenColumn();
                column.setTableId(sysGenTable.getTableId());
                column.setColumnName(columnName);
                String required = (String) tableMap.get("is_required");
                column.setIsRequired(required);
                column.setIsPk((String)tableMap.get("is_pk"));
                column.setSort(Integer.parseInt(tableMap.get("sort")+""));
                column.setColumnComment((String) tableMap.get("column_comment"));
                column.setIsIncrement((String) tableMap.get("is_increment"));
                String dbType = (String) tableMap.get("data_type");
                column.setColumnType(dbType);
                column.setJavaType(GenUtils.dbType2JavaType(dbType));
                column.setJavaField(StringUtils.toCamelCase(columnName));
                String insert = ("1".equals(required) || "int".equals(dbType) || ("varchar".equals(dbType) && !columnName.endsWith("_by"))) ? "1" : "0";
                column.setIsInsert(insert);
                column.setIsEdit(insert);
                column.setIsList(insert);
                String query = "varchar".equals(dbType) && !"remark".equals(columnName) && !columnName.endsWith("_by") ? "1" : "0";
                column.setIsQuery(query);
                column.setQueryType("1".equals(query) && "varchar".equals(dbType) ? "LIKE" : "EQ");
                String htmlType = null;
                if("1".equals(insert)) {
                    if ("varchar".equals(dbType)) {
                        htmlType = "input";
                    } else if ("tinyint".equals(dbType) || "char".equals(dbType)) {
                        htmlType = "radio";
                    } else if ("remark".equals(columnName)) {
                        htmlType = "textarea";
                    }
                }
                column.setHtmlType(htmlType);
                sysGenColumnMapper.insert(column);
            } else {
                // 修改字段
                String required = (String) tableMap.get("is_required");
                column.setIsRequired(required);
                column.setIsPk((String)tableMap.get("is_pk"));
                column.setSort(Integer.parseInt(tableMap.get("sort")+""));
                column.setColumnComment((String) tableMap.get("column_comment"));
                column.setIsIncrement((String) tableMap.get("is_increment"));
                String dbType = (String) tableMap.get("data_type");
                column.setColumnType(dbType);
                column.setJavaType(GenUtils.dbType2JavaType(dbType));
                column.setJavaField(StringUtils.toCamelCase(columnName));
                sysGenColumnMapper.updateById(column);
            }
        }

        List<String> dbColNames = tableColumns.stream().map(s -> (String)s.get("column_name")).collect(Collectors.toList());
        List<SysGenColumn> delCols = oldColumns.stream().filter(column -> !dbColNames.contains(column.getColumnName())).collect(Collectors.toList());
        if (StringUtils.isNotEmpty(delCols)) {
            sysGenColumnMapper.deleteGenTableColumns(delCols);
        }
    }

    private void generatorCode(String tableName, ZipOutputStream zip) {
        // 查询表信息
        SysGenTable table = sysGenTableMapper.selectGenTableByName(tableName);
        List<SysGenColumn> columns = table.getColumns();
        if(StringUtils.isEmpty(columns) || columns.size() <= 1) {
            sysGenColumnService.genColumnListByTableName(table.getTableId(), table.getTableName());
            table = sysGenTableMapper.selectGenTableById(table.getTableId());
        }
        // 设置主子表信息
        setSubTable(table);
        // 设置主键列信息
        setPkColumn(table);

        Map<String, Object> objectMap = freemarkerService.getObjectMap(table);

        // 获取模板列表
        List<String> templates = freemarkerService.getTemplateList(table.getTplCategory());
        for (String template : templates) {
            // 渲染模板
            StringWriter sw = new StringWriter();
            try {
                Template tpl = freemarkerConfiguration.getTemplate(template, GlobalConstant.UTF8);
                tpl.process(objectMap, sw);

                // 添加到zip
                zip.putNextEntry(new ZipEntry(freemarkerService.getFileName(template, table)));
                IOUtils.write(sw.toString(), zip, GlobalConstant.UTF8);
                IOUtils.closeQuietly(sw);
                zip.flush();
                zip.closeEntry();
            } catch (Exception e) {
                log.error("渲染模板失败，表名：" + table.getTableName(), e);
            }
        }
    }

    /**
     * 设置主键列信息
     *
     * @param table 业务表信息
     */
    public void setPkColumn(SysGenTable table) {
        for (SysGenColumn column : table.getColumns()) {
            if (column.isPk()) {
                table.setPkColumn(column);
                break;
            }
        }
        if (StringUtils.isNull(table.getPkColumn())) {
            table.setPkColumn(table.getColumns().get(0));
        }
        if (GenConstants.TPL_SUB.equals(table.getTplCategory())) {
            for (SysGenColumn column : table.getSubTable().getColumns()) {
                if (column.isPk()) {
                    table.getSubTable().setPkColumn(column);
                    break;
                }
            }
            if (StringUtils.isNull(table.getSubTable().getPkColumn())) {
                table.getSubTable().setPkColumn(table.getSubTable().getColumns().get(0));
            }
        }
    }

    /**
     * 设置主子表信息
     *
     * @param table 业务表信息
     */
    public void setSubTable(SysGenTable table) {
        String subTableName = table.getSubTableName();
        if (StringUtils.isNotEmpty(subTableName)) {
            table.setSubTable(sysGenTableMapper.selectGenTableByName(subTableName));
        }
    }
}
