package cn.morethank.open.admin.system.service.impl;

import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.service.DictService;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDictData;
import cn.morethank.open.admin.system.domain.SysDictType;
import cn.morethank.open.admin.system.mapper.SysDictDataMapper;
import cn.morethank.open.admin.system.mapper.SysDictTypeMapper;
import cn.morethank.open.admin.system.service.SysDictTypeService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 字典类型表 服务实现类
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Service
public class SysDictTypeServiceImpl extends ServiceImpl<SysDictTypeMapper, SysDictType> implements SysDictTypeService {

    private final SysDictTypeMapper sysDictTypeMapper;
    private final SysDictDataMapper dictDataMapper;
    private final DictService dictService;

    @Override
    public IPage<SysDictType> selectListPage(Page<SysDictType> page, LambdaQueryWrapper<SysDictType> query) {
        return sysDictTypeMapper.selectPage(page, query);
    }

    /**
     * 根据字典类型查询字典数据
     *
     * @param dictType 字典类型
     * @return 字典数据集合信息
     */
    @Override
    public List<SysDictData> selectDictDataByType(String dictType) {
        List<SysDictData> dictDatas = dictService.getDictCache(dictType);
        if (StringUtils.isNotEmpty(dictDatas)) {
            return dictDatas;
        }
        dictDatas = dictDataMapper.selectDictDataByType(dictType);
        if (StringUtils.isNotEmpty(dictDatas)) {
            dictService.setDictCache(dictType, dictDatas);
            return dictDatas;
        }
        return null;
    }

    @Override
    public int deleteDictTypeByIds(Long[] dictIds) {
        for (Long dictId : dictIds) {
            SysDictType dictType = getById(dictId);
            LambdaQueryWrapper<SysDictData> query = new LambdaQueryWrapper<>();
            query.eq(SysDictData::getDictType, dictType.getDictType());
            long useCount = dictDataMapper.selectCount(query);
            if (useCount > 0) {
                throw new ServiceException(String.format("%1$s已分配,不能删除", dictType.getDictName()));
            }
            sysDictTypeMapper.deleteById(dictId);
            dictService.removeDictCache(dictType.getDictType());
        }
        return dictIds.length;
    }

    @Override
    public boolean checkDictTypeUnique(SysDictType dict) {
        LambdaQueryWrapper<SysDictType> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysDictType::getDictType, dict.getDictType());
        if(dict.getDictId() != null && dict.getDictId() > 0L) {
            wrapper.ne(SysDictType::getDictId, dict.getDictId());
        }
        long count = sysDictTypeMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public void resetDictCache() {
        dictService.clearDictCache();
        loadingDictCache();
    }

    private void loadingDictCache() {
        SysDictData dictData = new SysDictData();
        dictData.setStatus("0");
        Map<String, List<SysDictData>> dictDataMap = dictDataMapper.selectDictDataList(dictData).stream().collect(Collectors.groupingBy(SysDictData::getDictType));
        for (Map.Entry<String, List<SysDictData>> entry : dictDataMap.entrySet()) {
            dictService.setDictCache(entry.getKey(), entry.getValue().stream().sorted(Comparator.comparing(SysDictData::getDictSort)).collect(Collectors.toList()));
        }
    }
}
