package cn.morethank.open.admin.system.service.impl;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.domain.TreeSelect;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDept;
import cn.morethank.open.admin.system.domain.SysRole;
import cn.morethank.open.admin.system.domain.SysUser;
import cn.morethank.open.admin.system.mapper.SysDeptMapper;
import cn.morethank.open.admin.system.mapper.SysRoleMapper;
import cn.morethank.open.admin.system.mapper.SysUserMapper;
import cn.morethank.open.admin.system.service.SysDeptService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 部门表 服务实现类
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Service
public class SysDeptServiceImpl extends ServiceImpl<SysDeptMapper, SysDept> implements SysDeptService {

    private final SysDeptMapper sysDeptMapper;
    private final SysUserMapper sysUserMapper;
    private final SysRoleMapper sysRoleMapper;

    @Override
    public boolean checkDeptNameUnique(SysDept dept) {
        LambdaQueryWrapper<SysDept> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysDept::getDeptName, dept.getDeptName()).eq(SysDept::getParentId, dept.getParentId())
                .eq(SysDept::getDelFlag, GlobalConstant.UNDELETED);
        if(dept.getDeptId() != null && dept.getDeptId() > 0L) {
            wrapper.ne(SysDept::getDeptId, dept.getDeptId());
        }
        long count = sysDeptMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public int deleteDeptById(Long deptId) {
        SysDept dept = new SysDept();
        dept.setDeptId(deptId);
        dept.setDelFlag(GlobalConstant.DELETED);
        return sysDeptMapper.updateById(dept);
    }

    @Override
    public boolean hasChildByDeptId(Long deptId) {
        LambdaQueryWrapper<SysDept> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysDept::getParentId, deptId).eq(SysDept::getDelFlag, GlobalConstant.UNDELETED);
        long count = sysDeptMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public boolean checkDeptExistUser(Long deptId) {
        LambdaQueryWrapper<SysUser> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysUser::getDeptId, deptId).eq(SysUser::getDelFlag, GlobalConstant.UNDELETED);
        long count = sysUserMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public int selectNormalChildrenDeptById(Long deptId) {
        return sysUserMapper.selectNormalChildrenDeptById(deptId);
    }

    @Override
    public List<SysDept> selectDeptList(SysDept dept) {
        return sysDeptMapper.selectDeptList(dept);
    }

    @Override
    public List<TreeSelect> selectDeptTreeList(SysDept dept) {
        List<SysDept> depts = selectDeptList(dept);
        return buildDeptTreeSelect(depts);
    }

    /**
     * 构建前端所需要下拉树结构
     *
     * @param depts 部门列表
     * @return 下拉树结构列表
     */
    @Override
    public List<TreeSelect> buildDeptTreeSelect(List<SysDept> depts) {
        List<SysDept> deptTrees = buildDeptTree(depts);
        return deptTrees.stream().map(TreeSelect::new).collect(Collectors.toList());
    }

    /**
     * 构建前端所需要树结构
     *
     * @param depts 部门列表
     * @return 树结构列表
     */
    @Override
    public List<SysDept> buildDeptTree(List<SysDept> depts) {
        List<SysDept> returnList = new ArrayList<SysDept>();
        List<Long> tempList = new ArrayList<Long>();
        for (SysDept dept : depts) {
            tempList.add(dept.getDeptId());
        }
        for (SysDept dept : depts) {
            // 如果是顶级节点, 遍历该父节点的所有子节点
            if (!tempList.contains(dept.getParentId())) {
                recursionFn(depts, dept);
                returnList.add(dept);
            }
        }
        if (returnList.isEmpty()) {
            returnList = depts;
        }
        return returnList;
    }

    @Override
    public List<Long> selectDeptListByRoleId(Long roleId) {
        SysRole role = sysRoleMapper.selectById(roleId);
        return sysDeptMapper.selectDeptListByRoleId(roleId, role.isDeptCheckStrictly());
    }

    /**
     * 递归列表
     */
    private void recursionFn(List<SysDept> list, SysDept t) {
        // 得到子节点列表
        List<SysDept> childList = getChildList(list, t);
        t.setChildren(childList);
        for (SysDept tChild : childList) {
            if (hasChild(list, tChild)) {
                recursionFn(list, tChild);
            }
        }
    }

    /**
     * 得到子节点列表
     */
    private List<SysDept> getChildList(List<SysDept> list, SysDept t) {
        List<SysDept> tlist = new ArrayList<SysDept>();
        Iterator<SysDept> it = list.iterator();
        while (it.hasNext()) {
            SysDept n = (SysDept) it.next();
            if (StringUtils.isNotNull(n.getParentId()) && n.getParentId().longValue() == t.getDeptId().longValue()) {
                tlist.add(n);
            }
        }
        return tlist;
    }

    /**
     * 判断是否有子节点
     */
    private boolean hasChild(List<SysDept> list, SysDept t) {
        return getChildList(list, t).size() > 0;
    }
}
